<?php

namespace Inspire_Labs\Empik_Woocommerce\Wp_Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Exception;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Client;
use Inspire_Labs\Empik_Woocommerce\Api_Server\Ping_Export_Offers_Factory;
use Inspire_Labs\Empik_Woocommerce\Api_Server\Ping_Export_Products_Factory;
use Inspire_Labs\Empik_Woocommerce\Api_Server\Ping_Import_Orders_Factory;
use Inspire_Labs\Empik_Woocommerce\App;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use WC_Admin_API_Keys;
use WC_Admin_Settings;
use WC_Admin_Webhooks;
use WC_Settings_Page;

/**
 * Class WC_Settings_Static_Factory
 */
class Settings extends WC_Settings_Page {

	const AJAX_BUTTON_TYPE_PRODUCTS        = 0;
	const AJAX_BUTTON_TYPE_OFFERS          = 1;
	const AJAX_BUTTON_TYPE_ORDERS          = 2;
	const AJAX_BUTTON_TYPE_CONNECTION_TEST = 3;
	const FALSE_SAVE_BUTTON                = 4;
	const OPTION_TRUE_VALUE                = '1';


	static $prevent_duplicate = array();


	/**
	 * Settings constructor.
	 */
	public function __construct() {
		$this->id    = Settings_Ids::ID;
		$this->label = esc_html__( 'Empik', 'empik-for-woocommerce' );

		add_action(
			'woocommerce_settings_save_' . $this->id,
			array( $this, 'save' )
		);

		parent::__construct();
	}


	/**
	 * Adds settings to Woocommerce.
	 */
	public function init() {
	}

	/**
	 * Get sections.
	 *
	 * @return array
	 */
	public function get_sections() {
		$sections = array(
			''             => esc_html__( 'Connection', 'empik-for-woocommerce' ),
			'products'     => esc_html__( 'Products', 'empik-for-woocommerce' ),
			'offers'       => esc_html__( 'Offers', 'empik-for-woocommerce' ),
			'orders'       => esc_html__( 'Orders', 'empik-for-woocommerce' ),
			'cron-actions' => esc_html__( 'CRON actions', 'empik-for-woocommerce' ),
		);

		return apply_filters( 'woocommerce_get_sections_' . $this->id, $sections );
	}


	/**
	 * Adds custom sections to the Empik WooCommerce settings tab.
	 *
	 * Appends 'Settings' and 'Help' sections to the existing sections array
	 * for the plugin's configuration interface.
	 *
	 * @param array $sections Existing sections array.
	 * @return array Modified sections array with new entries.
	 */
	public function get_sections_empik_woocommerce( $sections ): array {
		$sections['settings'] = esc_html__( 'Settings', 'empik-for-woocommerce' );
		$sections['help']     = esc_html__( 'Help', 'empik-for-woocommerce' );

		return $sections;
	}

	/**
	 * Adds the Empik WooCommerce settings tab to WooCommerce settings.
	 *
	 * Appends a new tab for the plugin's configuration to the existing
	 * WooCommerce settings tabs array.
	 *
	 * @param array $settings_tabs Existing WooCommerce settings tabs.
	 * @return array Modified settings tabs array with new Empik tab.
	 */
	public function add_settings_tab( array $settings_tabs ): array {

		$settings_tabs['settings_tab_empik_woocommerce'] = esc_html__( 'empik_woocommerce', 'empik-for-woocommerce' );

		return $settings_tabs;
	}

	/**
	 * Creates connection settings configuration array for Empik integration.
	 *
	 * Defines form fields for API key, shop ID, environment selection (production/testing),
	 * and connection test button with their respective types and properties.
	 *
	 * @return array Connection settings configuration array.
	 */
	public function create_settings_connection(): array {
		return array_merge(
			array(

				Settings_Ids::SETTINGS_ID_API_KEY     => array(
					'name' => esc_html__( 'API key', 'empik-for-woocommerce' ),
					'type' => 'text',
					'desc' => '',
					'id'   => Settings_Ids::SETTINGS_ID_API_KEY,
				),

				Settings_Ids::SETTINGS_ID_SHOP_ID     => array(
					'name' => esc_html__( 'Shop ID', 'empik-for-woocommerce' ),
					'type' => 'text',
					'desc' => '',
					'id'   => Settings_Ids::SETTINGS_ID_SHOP_ID,
				),

				Settings_Ids::SETTINGS_ID_ENVIRONMENT => array(
					'name'    => esc_html__(
						'Select environment',
						'empik-for-woocommerce'
					),
					'type'    => 'radio',
					'options' => array(
						Settings_Ids::SETTINGS_ID_ENVIRONMENT_PROD    => esc_html__(
							'Production',
							'empik-for-woocommerce'
						),
						Settings_Ids::SETTINGS_ID_ENVIRONMENT_TESTING => esc_html__(
							'Testing',
							'empik-for-woocommerce'
						),
					),
					'default' => esc_html__( 'Production', 'empik-for-woocommerce' ),
					'desc'    => '',
					'id'      => Settings_Ids::SETTINGS_ID_ENVIRONMENT,
				),
				Settings_Ids::SETTINGS_ID_TEST_CONNECTION_BTN => array(
					'name'    => '',
					'type'    => 'text',
					'default' => esc_html__( 'Test connection', 'empik-for-woocommerce' ),
					'desc'    => '',
					'id'      => Settings_Ids::SETTINGS_ID_TEST_CONNECTION_BTN,
				),
			)
		);
	}

	/**
	 * Creates an AJAX button configuration for admin settings.
	 *
	 * Generates button settings for different operations (products export, offers export,
	 * orders import, or save changes) and registers the corresponding admin field action
	 * to render the button with spinner and nonce protection.
	 *
	 * @param int $type The button type constant (products, offers, orders, or save).
	 * @return array Button configuration array for settings.
	 * @throws Exception When an incorrect button type parameter is provided.
	 */
	private function create_ajax_button( int $type ): array {
		// todo $custom_button_ids powinny być pobierne z jednego miejsca w projekcie.
		if ( self::AJAX_BUTTON_TYPE_PRODUCTS === $type ) {
			$custom_button_id = Plugin::APP_PREFIX . '_button_export_products';
			$settings_id      = Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_BTN;
			$title            = esc_html__(
				'Export products now',
				'empik-for-woocommerce'
			);
			$label            = '';
		} elseif ( self::AJAX_BUTTON_TYPE_OFFERS === $type ) {
			$custom_button_id = Plugin::APP_PREFIX . '_button_export_offers';
			$title            = esc_html__(
				'Export offers now',
				'empik-for-woocommerce'
			);
			$label            = '';
			$settings_id      = Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_BTN;
		} elseif ( self::AJAX_BUTTON_TYPE_ORDERS === $type ) {
			$custom_button_id = Plugin::APP_PREFIX . '_button_import_orders';
			$title            = esc_html__(
				'Import orders now',
				'empik-for-woocommerce'
			);
			$label            = '';
			$settings_id      = Settings_Ids::SETTINGS_ID_EXPORT_ORDERS_BTN;
		} elseif ( self::FALSE_SAVE_BUTTON === $type ) {
			$custom_button_id = Plugin::APP_PREFIX . '_button_connection_save';
			$title            = esc_html__( 'Save changes', 'empik-for-woocommerce' );
			$label            = '';
			$settings_id      = Settings_Ids::SETTINGS_ID_TEST_CONNECTION_SAVE;

		} else {
			throw new Exception( 'get_export_button() - incorrect parameter' );
		}

		add_action(
			'woocommerce_admin_field_' . $custom_button_id,
			function ( array $data ) use ( $custom_button_id ) {
				$type = $data['type'];
				if ( isset( self::$prevent_duplicate[ $type ] )
					&& self::$prevent_duplicate[ $type ]
				) {
					return;
				}
				if ( $custom_button_id === $type ) {
					?>
						<tr valign="top">
							<th scope="row" class="titledesc">
								<label for="<?php echo esc_attr( $custom_button_id ); ?>">
									<?php echo esc_attr( $data['name'] ); ?></label>
							</th>
							<td class="forminp forminp-text">
								<button class="button il-empik-btn"
										style="margin-top:20px;margin-right:2500px;"
										id="<?php echo esc_attr( $custom_button_id ); ?>">
									<?php echo esc_html( $data['title'] ); ?>
								</button>
								<p class="empik-preloader-wrapper <?php echo esc_attr( $custom_button_id ); ?>">
								<span style="float:unset;margin-top: 20px;visibility: hidden;" class="empik-ajax-run spinner is-active"></span>
								<span class="empik-ajax-procent" style="float:unset;margin-top: 20px;position:absolute"></span>
								<input id="empik-export-nonce" type="hidden" value="<?php echo esc_attr( wp_create_nonce( 'empik-export-nonce' ) ); ?>">
								</p>
							</td>
						</tr>
						<?php
				}
				self::$prevent_duplicate[ $type ] = true;
			},
			10
		);

		return array(
			$settings_id => array(
				'name'  => $label,
				'title' => $title,
				'type'  => $custom_button_id,
				'id'    => $settings_id,
			),
		);
	}

	/**
	 * Creates API URLs display configuration for cron endpoints.
	 *
	 * Generates a settings field that displays the API endpoints for offers export,
	 * orders import, and products export, along with a link to the CRON configuration
	 * manual for automatic task setup.
	 *
	 * @return array API URLs settings configuration array.
	 */
	private function create_api_urls(): array {

		$custom_button_id = Plugin::APP_PREFIX . '_api_urls';

		add_action(
			'woocommerce_admin_field_' . $custom_button_id,
			function ( array $data ) use ( $custom_button_id ) {

				$offer_url    = ( new Ping_Export_Offers_Factory() )->create_service()->get_url();
				$orders_url   = ( new Ping_Import_Orders_Factory() )->create_service()->get_url();
				$products_url = ( new Ping_Export_Products_Factory() )->create_service()->get_url();

				$type = $data['type'];
				if ( isset( self::$prevent_duplicate[ $type ] )
					&& self::$prevent_duplicate[ $type ]
				) {
					return;
				}
				if ( $custom_button_id === $type ) {
					?>
							<br>
						<tr style="margin-top: 20px">
							<th scope="row" class="titledesc">
								<h3 class="empik_server_cron_endpoints">
									<?php echo esc_attr( $data['name'] ); ?>
								</h3>
								<h3 class="empik_server_wp_cron">
									<?php echo esc_html__( 'Attention:', 'empik-for-woocommerce' ); ?>
								</h3>
							</th>
							<td class="forminp forminp-text">
								<ul class="empik_server_cron_endpoints">
									<li><?php echo esc_url( $offer_url ); ?></li>
									<li><?php echo esc_url( $orders_url ); ?></li>
									<li><?php echo esc_url( $products_url ); ?></li>
								</ul>
								<p class="empik_server_cron_endpoints">
									<?php
									printf(
										'%1s <b><a href="%2s" target="_blank">%3s</a></b>',
										esc_html__( 'Information about configuring automatic actions (CRON) can be found in our', 'empik-for-woocommerce' ),
										esc_url( 'https://www.pomoc.empikplace.com/portal/pl/kb/articles/zadania-cron-22-3-2024' ),
										esc_html__( 'Manual', 'empik-for-woocommerce' )
									);
									?>
								</p>
								<p class="empik_server_wp_cron">
									<?php
									printf(
										'%1s<br>%2s <b>%3s</b><br>%4s<br>%5s',
										esc_html__( 'This option uses WP-Cron, which runs in the background and usually works well for most stores.', 'empik-for-woocommerce' ),
										esc_html__( 'However, it is worth noting that when WP-Cron is used in stores with low-traffic', 'empik-for-woocommerce' ),
										esc_html__( 'it may sometimes miss scheduled tasks.', 'empik-for-woocommerce' ),
										esc_html__( 'To avoid such situations, you can use a more reliable solution: Cron jobs set directly on the server with your hosting provider.', 'empik-for-woocommerce' ),
										esc_html__( '(Select „No” to see the configuration URL.)', 'empik-for-woocommerce' )
									);
									?>
								</p>
							</td>
						</tr>
						<?php
				}
				self::$prevent_duplicate[ $type ] = true;
			},
			10
		);

		$settings_id = Settings_Ids::SETTINGS_ID_API_URLS;

		return array(
			$settings_id => array(
				'name'  => esc_html__( 'Cron API endpoints:', 'empik-for-woocommerce' ),
				'title' => 'title',
				'type'  => $custom_button_id,
				'id'    => $settings_id,
			),
		);
	}


	/**
	 * Creates product export settings configuration array.
	 *
	 * Generates settings fields for product export including description with
	 * mapping wizard instructions, checkbox for exporting all products, and
	 * an AJAX export button for immediate product synchronization.
	 *
	 * @return array Product export settings configuration array.
	 */
	public function create_settings_products(): array {

		return array_merge(
			array(
				// Send all products
				Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_DESCRIPTION => array(
					'title'             => '',
					'type'              => 'text',
					'desc'              => sprintf(
						"<span class='empik-pexport-help'>%s</span><br>
                                                    <ol class='empik-pexport-help'>
                                                    <li>%s <a target='_blank' rel='noreferrer nofollow' href='%s'>%s</a> %s</li>
                                                    <li>%s <a target='_blank' rel='noreferrer nofollow' href='%s'>%s</a></li>
                                                    </ol>
                                                    <br>
                                                    <span class='empik-pexport-help empik-pexport-help-end'>%s <a target='_blank' rel='noreferrer nofollow' href='%s'>%s</a></span>",
						esc_html__( 'After importing the products, go to the EmpikPlace Sales Panel and create a mapping for the uploaded file using the mapping wizard', 'empik-for-woocommerce' ),
						esc_html__( 'After import, in the Seller Panel, go to', 'empik-for-woocommerce' ),
						esc_url( 'https://marketplace.empik.com/mmp/shop/product/import/history' ),
						esc_html__( 'Catalog -> Product Imports', 'empik-for-woocommerce' ),
						esc_html__( 'and get the last import.', 'empik-for-woocommerce' ),
						esc_html__( 'The file should be mapped using the mapping wizard located in', 'empik-for-woocommerce' ),
						esc_url( 'https://marketplace.empik.com/mmp/shop/catalog/mapping/wizard' ),
						esc_html__( 'Settings -> Directory Mapping -> Use Mapping Wizard button', 'empik-for-woocommerce' ),
						esc_html__( 'Only then will subsequent product files be processed correctly. Instructions for using the mapping wizard can be found', 'empik-for-woocommerce' ),
						esc_url( 'https://www.pomoc.empikplace.com/portal/pl/kb/articles/woocommerce#Produkty' ),
						esc_html__( 'HERE', 'empik-for-woocommerce' )
					),
					'custom_attributes' => array(
						'required' => false,
						'hidden'   => true,
					),
					'id'                => Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_DESCRIPTION,
				),
			),
			array(
				'type' => 'sectionend',
				'id'   => Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_DESCRIPTION,
			),
			array(
				// Send all products.
				Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_ALL => array(
					'name' => '',
					'type' => 'checkbox',
					'desc' => esc_html__( 'Export all products?', 'empik-for-woocommerce' ),
					'id'   => Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_ALL,
				),
			),
			$this->create_ajax_button( self::AJAX_BUTTON_TYPE_PRODUCTS )
		);
	}


	/**
	 * Creates offer export settings configuration array.
	 *
	 * Generates comprehensive settings for offer synchronization including automatic sync options,
	 * ID type selection (SKU/EAN), stock control, pricing configuration (dedicated prices or
	 * multiplier-based), logistics class and product state options with bulk change capabilities.
	 *
	 * @return array Offer export settings configuration array.
	 */
	public function create_settings_offers(): array {

		return array_merge(
			array(
				array(
					'title' => '',
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'empik_options_1',
				),

				Settings_Ids::SETTINGS_ID_EXPORT_OFFERS => array(
					'name'    => esc_html__(
						'Synchronize offers automatically witch Empik Marketplace?',
						'empik-for-woocommerce'
					),
					'type'    => 'radio',
					'options' => array(
						'1' => esc_html__( 'Yes', 'empik-for-woocommerce' ),
						'0' => esc_html__( 'No', 'empik-for-woocommerce' ),
					),
					'default' => '1',
					'desc'    => esc_html__( 'Synchronize offers automatically witch Empik Marketplace?', 'empik-for-woocommerce' ),
					'id'      => Settings_Ids::SETTINGS_ID_EXPORT_OFFERS,
				),

				Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ID_TYPE => array(
					'name'    => esc_html__( 'Offer ID type', 'empik-for-woocommerce' ),
					'type'    => 'radio',
					'options' => array(
						'sku' => 'SKU',
						'ean' => 'EAN',

					),
					'default' => 'sku',
					'desc'    => esc_html__( 'Offer ID type', 'empik-for-woocommerce' ),
					'id'      => Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ID_TYPE,
				),

				Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ID_FIELD => array(
					'name'    => esc_html__( 'EAN metafield key', 'empik-for-woocommerce' ),
					'type'    => 'text',
					'default' => '',
					'desc'    => esc_html__( 'EAN metafield key', 'empik-for-woocommerce' ),
					'id'      => Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ID_FIELD,
				),

				Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_STOCK_CONTROL => array(
					'name'              => esc_html__( 'If qty of product will be less or equal to value of this field then 0 qty will be set in export file for such product.', 'empik-for-woocommerce' ),
					'type'              => 'number',
					'default'           => '0',
					'desc'              => esc_html__( 'If qty of product will be less or equal to value of this field then 0 qty will be set in export file for such product.', 'empik-for-woocommerce' ),
					'id'                => Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_STOCK_CONTROL,
					'custom_attributes' => array(
						'step' => '1',
						'min'  => '0',
					),
				),

				array(
					'type' => 'sectionend',
					'id'   => 'empik_options_1',
				),

			),
			array(
				array(
					'title' => '',
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'empik_options_2',
				),

				Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_FROM => array(
					'name'    => esc_html__( 'Choose source of prices for products', 'empik-for-woocommerce' ),
					'type'    => 'radio',
					'options' => array(
						'1' => esc_html__( 'Dedicated prices for offers', 'empik-for-woocommerce' ),
						'0' => esc_html__( 'Prices of offers set up from multiplier', 'empik-for-woocommerce' ),
					),
					'default' => '1',
					'desc'    => esc_html__( 'Choose source of prices for products', 'empik-for-woocommerce' ),
					'id'      => Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_FROM,
					'class'   => 'empik_source_prices',
					'css'     => '',
				),

				Settings_Ids::SETTINGS_ID_OFFERS_PRICE_MULTIPLIER => array(
					'name'              => esc_html__( 'Increase the price by a percentage (%)', 'empik-for-woocommerce' ),
					'type'              => 'number',
					'desc'              => esc_html__( 'Increase the price by a percentage (%)', 'empik-for-woocommerce' ),
					'id'                => Settings_Ids::SETTINGS_ID_OFFERS_PRICE_MULTIPLIER,
					'custom_attributes' => array(
						'step'  => '0.01',
						'class' => 'empik_offers_input',
					),
				),
				Settings_Ids::SETTINGS_ID_OFFERS_ADD_TO_PRICE     => array(
					'name'              => esc_html__( 'Add value to the price', 'empik-for-woocommerce' ),
					'type'              => 'number',
					'desc'              => esc_html__( 'Add value to the price', 'empik-for-woocommerce' ),
					'id'                => Settings_Ids::SETTINGS_ID_OFFERS_ADD_TO_PRICE,
					'custom_attributes' => array(
						'step'  => '0.01',
						'class' => 'empik_offers_input',
					),
				),

				// Send all offers.
				Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ALL => array(
					'name' => '',
					'type' => 'checkbox',
					'desc' => esc_html__( 'Export offers for all products?', 'empik-for-woocommerce' ) .
						'<br><button class="button il-empik-btn" style="margin-top:20px;" id="empik_woocommerce_button_export_offers">
							' . esc_html__( 'Export offers now', 'empik-for-woocommerce' ) . '</button><p class="empik-preloader-wrapper empik_woocommerce_button_export_offers">
                                <span style="float:unset;margin-top: 20px;visibility: hidden;" class="empik-ajax-run spinner"></span>
                                <span class="empik-ajax-procent" style="float:unset;margin-top: 20px;position:absolute"></span>
                                <span id="empik-export-nonce" style="display:none;" data-id="' . esc_attr( wp_create_nonce( 'empik-export-nonce' ) ) . '"></span>
								</p>',
					'id'   => Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ALL,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'empik_options_2',
				),

			),
			array(
				array(
					'title' => '',
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'empik_options_3',
				),

				Settings_Ids::SETTINGS_ID_OFFERS_ADD_LC_TO_FILE     => array(
					'name' => '',
					'type' => 'checkbox',
					'desc' => sprintf(
						'%1s<br><br><i>%2s</i><br>%3s',
						esc_html__( 'Yes', 'empik-for-woocommerce' ),
						esc_html__( 'Attention! Activation of this option will result in the dispatch of the default logistics class (Medium code=2) or the one selected on the product card (Shipping tab).', 'empik-for-woocommerce' ),
						'<a href="?page=ilabs_empik&tab=empik_csv_1" 
                                                class="button"
                                                style="margin-top:20px;" 
                                                id="empik_woocommerce_button_csv"
                                                >'
						. esc_html__( 'Bulk change of logistics class', 'empik-for-woocommerce' ) .
						'</a>'
					),
					'id'   => Settings_Ids::SETTINGS_ID_OFFERS_ADD_LC_TO_FILE,
					'css'  => 'margin-top:10px',
				),

				array(
					'type' => 'sectionend',
					'id'   => 'empik_options_3',
				),

				array(
					'title' => '',
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'empik_options_4',
				),

				Settings_Ids::SETTINGS_ID_OFFERS_ADD_PS_TO_FILE     => array(
					'name' => '',
					'type' => 'checkbox',
					'desc' => sprintf(
						'%1s<br><br><i>%2s<b>%3s</b>, <b>%4s</b>, <b>%5s</b></i><br><i>%6s</i><br>%7s',
						esc_html__( 'Yes', 'empik-for-woocommerce' ),
						esc_html__( 'Attention! This option allow to set the product state like ', 'empik-for-woocommerce' ),
						esc_html__( 'new', 'empik-for-woocommerce' ),
						esc_html__( 'used', 'empik-for-woocommerce' ),
						esc_html__( 'refurbished or with damaged package', 'empik-for-woocommerce' ),
						esc_html__( 'Activation will result in the dispatch of the default product state (New code=11) or the one selected on the product card (Main tab).', 'empik-for-woocommerce' ),
						'<a href="?page=ilabs_empik&tab=empik_csv_2" 
                                    class="button"
                                    style="margin-top:20px;" 
                                    id="empik_woocommerce_button_csv_2"
                                    >'
						. esc_html__( 'Bulk change of product states', 'empik-for-woocommerce' ) .
						'</a>'
					),
					'id'   => Settings_Ids::SETTINGS_ID_OFFERS_ADD_PS_TO_FILE,
					'css'  => 'margin-top:10px',
				),

				array(
					'type' => 'sectionend',
					'id'   => 'empik_options_4',
				),
			)
		);
	}


	/**
	 * Creates order management settings configuration array.
	 *
	 * Generates settings for automatic order confirmation and automatic order
	 * downloading from Empik Marketplace with yes/no radio button options.
	 *
	 * @return array Order management settings configuration array.
	 */
	public function create_settings_orders(): array {
		return array_merge(
			array(
				Settings_Ids::SETTINGS_ID_CONFIRM_ORDERS_AUTOMATICALLY => array(
					'name'    => esc_html__(
						'Confirm orders automatically (default - yes)',
						'empik-for-woocommerce'
					),
					'type'    => 'radio',
					'options' => array(
						'1' => esc_html__( 'Yes', 'empik-for-woocommerce' ),
						'0' => esc_html__( 'No', 'empik-for-woocommerce' ),
					),
					'default' => '1',
					'desc'    => '',
					'id'      => Settings_Ids::SETTINGS_ID_CONFIRM_ORDERS_AUTOMATICALLY,
				),

				Settings_Ids::SETTINGS_ID_DOWNLOAD_ORDERS_AUTOMATICALLY => array(
					'name'    => esc_html__(
						'Automatically download orders from Empik Marketplace',
						'empik-for-woocommerce'
					),
					'type'    => 'radio',
					'options' => array(
						'1' => esc_html__( 'Yes', 'empik-for-woocommerce' ),
						'0' => esc_html__( 'No', 'empik-for-woocommerce' ),
					),
					'default' => '1',
					'desc'    => '',
					'id'      => Settings_Ids::SETTINGS_ID_DOWNLOAD_ORDERS_AUTOMATICALLY,
				),

			)
		);
	}

	/**
	 * Gets settings configuration based on the current section.
	 *
	 * Returns connection settings for empty section parameter, otherwise
	 * returns an empty array for other sections.
	 *
	 * @param string $current_section The current settings section identifier.
	 * @return array Settings configuration array for the specified section.
	 */
	public function get_settings( $current_section = '' ): array {
		if ( empty( $current_section ) ) {
			return $this->create_settings_connection();
		}

		return array();
	}


	/**
	 * Form method.
	 *
	 * @param string $method Method name.
	 *
	 * @return string
	 * @deprecated 3.4.4
	 */
	public function form_method( $method ) {
		return 'post';
	}

	/**
	 * Outputs the appropriate settings fields based on the current section.
	 *
	 * Renders different settings configurations for products, offers, orders,
	 * cron-actions sections, or connection settings with custom CSS styling
	 * for the default section.
	 *
	 * @return void
	 */
	public function output() {
		global $current_section;

		if ( 'products' === $current_section ) {
			$settings = $this->create_settings_products();
			WC_Admin_Settings::output_fields( $settings );
		} elseif ( 'offers' === $current_section ) {
			$settings = $this->create_settings_offers();
			WC_Admin_Settings::output_fields( $settings );
		} elseif ( 'orders' === $current_section ) {
			$settings = $this->create_settings_orders();
			WC_Admin_Settings::output_fields( $settings );
		} elseif ( 'cron-actions' === $current_section ) {
			$settings = $this->create_settings_cron();
			WC_Admin_Settings::output_fields( $settings );
		} else {
			echo '<style>
                    /*.woocommerce-save-button {
                        display: none !important;
                    }*/
                    label[for="empik_woocommerce_set_environment"] {
                        display: block;
                        padding-top: 30px;
                    }                    
                    #empik_woocommerce_set_shop_id_key {
                        margin-top: 20px;
                    }
                    #empik_woocommerce_set_api_key {
                        margin-top: 20px;
                    }                    
                    .empik-preloader-wrapper.empik_woocommerce_button_connection_save {
                        display: none !important;
                    }
                </style>';
			$settings = $this->create_settings_connection();
			WC_Admin_Settings::output_fields( $settings );
		}
	}


	/**
	 * Saves settings fields based on the current section.
	 *
	 * Determines the appropriate settings configuration for the current section,
	 * saves the fields using WooCommerce admin settings, and triggers section-specific
	 * update actions when applicable.
	 *
	 * @return void
	 */
	public function save() {
		global $current_section;

		if ( 'products' === $current_section ) {
			$settings = $this->create_settings_products();
		} elseif ( 'offers' === $current_section ) {
			$settings = $this->create_settings_offers();
		} elseif ( 'orders' === $current_section ) {
			$settings = $this->create_settings_orders();
		} elseif ( 'cron-actions' === $current_section ) {
			$settings = $this->create_settings_cron();
		} else {
			$settings = $this->create_settings_connection();
		}

		WC_Admin_Settings::save_fields( $settings );

		if ( $current_section ) {
			do_action(
				'woocommerce_update_options_' . $this->id . '_'
						. $current_section
			);
		}
	}



	/**
	 * Creates cron settings configuration array.
	 *
	 * Generates settings for choosing between WordPress CRON automation
	 * (with predefined schedules) or manual server CRON configuration,
	 * and includes API URLs display for endpoint reference.
	 *
	 * @return array Cron settings configuration array.
	 */
	public function create_settings_cron(): array {

		return array_merge(
			array(
				Settings_Ids::SETTINGS_ID_CRON_TYPE => array(
					'name'    => esc_html__( 'Do you want use WP CRON to automatically manage the scheduling of these tasks for your account, or would you prefer to configure them yourself on your server?', 'empik-for-woocommerce' ),
					'type'    => 'radio',
					'options' => array(
						'1' => esc_html__( 'Yes, set WP CRON jobs (schedule: orders - 10 min, offers - 1 h, products - 8 h)', 'empik-for-woocommerce' ),
						'0' => esc_html__( 'No, I will set up CRON jobs on the server myself', 'empik-for-woocommerce' ),
					),
					'default' => '0',
					'desc'    => '',
					'id'      => Settings_Ids::SETTINGS_ID_CRON_TYPE,
				),

			),
			$this->create_api_urls()
		);
	}
}
