<?php

namespace Inspire_Labs\Empik_Woocommerce\Product;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


use Inspire_Labs\Empik_Woocommerce\Api_Client\Api_Factory;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Ids;


/**
 * Product_Settings
 */
class Product_Settings {


	/**
	 * Initializes the product settings hooks and actions.
	 *
	 * Sets up WordPress action hooks for adding custom fields to WooCommerce
	 * product edit screens, including logistic class settings, pricing options
	 * for variations, and product state fields. Also registers save handlers
	 * for the custom fields.
	 */
	public function init() {
		add_action( 'woocommerce_product_options_shipping', array( $this, 'logistic_class_product_tab' ), 10, 0 );
		add_action( 'woocommerce_admin_process_product_object', array( $this, 'logistic_class_product_tab_save' ), 10, 1 );

		add_action( 'woocommerce_variation_options_pricing', array( $this, 'empik_product_price_settings_variation' ), 10, 3 );
		add_action( 'woocommerce_product_options_pricing', array( $this, 'empik_product_price_settings' ) );

		add_action( 'woocommerce_save_product_variation', array( $this, 'empik_save_variation_meta' ), 10, 2 );

		add_action( 'woocommerce_product_options_general_product_data', array( $this, 'product_states_variable' ) );
	}


	/**
	 * Displays the logistic class selection field on product edit screen.
	 *
	 * Renders a dropdown select field in the WooCommerce product shipping tab
	 * allowing users to choose the appropriate Empik logistic class for the
	 * product. Uses predefined options with translated labels and numeric codes.
	 */
	public function logistic_class_product_tab() {
		global $post;

		$options = array(
			'2'  => esc_html__( 'Medium (kod = 2)', 'empik-for-woocommerce' ),
			'1'  => esc_html__( 'Small (kod = 1)', 'empik-for-woocommerce' ),
			'3'  => esc_html__( 'Big (kod = 3)', 'empik-for-woocommerce' ),
			'0'  => esc_html__( 'For selected offers (kod = 0)', 'empik-for-woocommerce' ),
			'4'  => esc_html__( 'For selected offers (kod = 4)', 'empik-for-woocommerce' ),
			'5'  => esc_html__( 'For selected offers (kod = 5)', 'empik-for-woocommerce' ),
			'6'  => esc_html__( 'For selected offers (kod = 6)', 'empik-for-woocommerce' ),
			'7'  => esc_html__( 'For selected offers (kod = 7)', 'empik-for-woocommerce' ),
			'8'  => esc_html__( 'For selected offers (kod = 8)', 'empik-for-woocommerce' ),
			'9'  => esc_html__( 'For selected offers (kod = 9)', 'empik-for-woocommerce' ),
			'10' => esc_html__( 'For selected offers (kod = 10)', 'empik-for-woocommerce' ),
			'11' => esc_html__( 'For selected offers (kod = 11)', 'empik-for-woocommerce' ),
		);

		woocommerce_wp_select(
			array(
				'id'      => 'empik_logistic_klass',
				'label'   => esc_html__( 'Logistic class Empik', 'empik-for-woocommerce' ),
				'options' => $options,
				'value'   => get_post_meta( $post->ID, '_empik_logistic_klass', true ),
			)
		);
	}


	/**
	 * Saves Empik-specific product metadata when product is updated.
	 *
	 * Processes and saves custom Empik fields including logistic class, product
	 * state, pricing information, and sale date ranges. Handles both simple and
	 * variable product types with appropriate validation and sanitization.
	 *
	 * @param WC_Product $product The WooCommerce product object being saved.
	 */
	public function logistic_class_product_tab_save( $product ) {

		if ( isset( $_POST['product-type'] ) && 'variable' === $_POST['product-type'] ) {
			if ( isset( $_POST['empik_product_state_all_variants'] ) && ( (int) $_POST['empik_product_state_all_variants'] > 0 ) ) {
				$product->update_meta_data( '_empik_product_state_all_variants', sanitize_text_field( $_POST['empik_product_state_all_variants'] ) );
			} else {
				$product->delete_meta_data( '_empik_product_state_all_variants' );
			}
		}

		if ( isset( $_POST['empik_product_state'] ) ) {
			$product->update_meta_data( '_empik_product_state', sanitize_text_field( $_POST['empik_product_state'] ) );
		}

		if ( isset( $_POST['empik_logistic_klass'] ) ) {
			$product->update_meta_data( '_empik_logistic_klass', sanitize_text_field( $_POST['empik_logistic_klass'] ) );
		}

		if ( isset( $_POST['empik_price'] ) && ! empty( $_POST['empik_price'] ) ) {
			$product->update_meta_data( '_empik_price', sanitize_text_field( $_POST['empik_price'] ) );
		} else {
			$product->update_meta_data( '_empik_price', '' );
		}

		if ( isset( $_POST['empik_price_sale'] ) && ! empty( $_POST['empik_price_sale'] ) ) {
			$product->update_meta_data( '_empik_price_sale', sanitize_text_field( $_POST['empik_price_sale'] ) );
		} else {
			$product->update_meta_data( '_empik_price_sale', '' );
		}

		if ( isset( $_POST['empik_sale_date_start'] ) && ! empty( $_POST['empik_sale_date_start'] ) ) {
			$date_user_format_start = sanitize_text_field( $_POST['empik_sale_date_start'] );
			$product->update_meta_data( '_empik_sale_date_start', sanitize_text_field( $_POST['empik_sale_date_start'] ) );
		} else {
			$product->update_meta_data( '_empik_sale_date_start', '' );
		}

		if ( isset( $_POST['empik_sale_date_end'] ) && ! empty( $_POST['empik_sale_date_end'] ) ) {
			$date_user_format_end = sanitize_text_field( $_POST['empik_sale_date_end'] );
			$product->update_meta_data( '_empik_sale_date_end', sanitize_text_field( $_POST['empik_sale_date_end'] ) );
		} else {
			$product->update_meta_data( '_empik_sale_date_end', '' );
		}
	}


	/**
	 * Displays Empik pricing settings fields on product edit screen.
	 *
	 * Renders custom price fields for Empik including regular price, sale price,
	 * and sale date range inputs. Calculates automatic pricing based on global
	 * multipliers and adds JavaScript validation for sale price logic and date
	 * picker functionality.
	 */
	public function empik_product_price_settings() {

		global $post;

		$price_value   = get_post_meta( $post->ID, '_empik_price', true );
		$price_regular = get_post_meta( $post->ID, '_regular_price', true );

		if ( ! get_option( 'empik_woocommerce_set_export_offers_from' ) ) {

			$price_percent_multiplier = (float) Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_OFFERS_PRICE_MULTIPLIER );

			$price_add_value = (float) Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_OFFERS_ADD_TO_PRICE );

			if ( $price_percent_multiplier > 0 ) {
				$price_value = (float) $price_regular + ( (float) $price_regular * ( $price_percent_multiplier / 100 ) );
			}

			if ( $price_add_value > 0 ) {
				$price_value = (float) $price_regular + (float) $price_add_value;
			}
		}

		woocommerce_wp_text_input(
			array(
				'id'        => 'empik_price',
				'class'     => 'short wc_input_price',
				'label'     => esc_html__( 'Price Empik', 'empik-for-woocommerce' ) . ' (' . get_woocommerce_currency_symbol() . ')',
				'data_type' => 'price',
				'value'     => $price_value,
			)
		);

		woocommerce_wp_text_input(
			array(
				'id'        => 'empik_price_sale',
				'class'     => 'short wc_input_price',
				'label'     => esc_html__( 'Sale price Empik', 'empik-for-woocommerce' ) . ' (' . get_woocommerce_currency_symbol() . ')',
				'data_type' => 'price',
				'value'     => get_post_meta( $post->ID, '_empik_price_sale', true ),
			)
		); ?>

		<p class="form-field empik_sale_dates_field inited inited_media_selector">
			<label for="empik_sale_date_start">
				<?php echo esc_html__( 'Dates of sale', 'empik-for-woocommerce' ); ?>
			</label>
			<input type="text" class="short EmpikHasDatepicker" style=""
					name="empik_sale_date_start"
					id="empik_sale_date_start"
					placeholder="<?php echo esc_html__( 'From… YYYY-MM-DD', 'empik-for-woocommerce' ); ?>"
					maxlength="10"
					pattern="[0-9]{4}-(0[1-9]|1[012])-(0[1-9]|1[0-9]|2[0-9]|3[01])"
					value="<?php echo esc_attr( get_post_meta( $post->ID, '_empik_sale_date_start', true ) ); ?>"
			>
		</p>
		<p class="form-field empik_sale_dates_field inited inited_media_selector">
			<input type="text" class="short EmpikHasDatepicker" style=""
					name="empik_sale_date_end"
					id="empik_sale_date_end"
					placeholder="<?php echo esc_html__( 'To… YYYY-MM-DD', 'empik-for-woocommerce' ); ?>"
					maxlength="10"
					pattern="[0-9]{4}-(0[1-9]|1[012])-(0[1-9]|1[0-9]|2[0-9]|3[01])"
					value="<?php echo esc_attr( get_post_meta( $post->ID, '_empik_sale_date_end', true ) ); ?>"
			>
			<span class="description">
			<a href="#" class="empik_price_reset">
				<?php echo esc_html__( 'Reset Empik settings', 'empik-for-woocommerce' ); ?>
			</a>
		</span>
		</p>
		<?php
		$api_factory    = new Api_Factory();
		$api            = $api_factory->create_service();
		$product_states = $api->of61_get_product_states();

		if ( ! empty( $product_states ) && is_array( $product_states ) ) {
			if ( 'yes' === get_option( 'empik_woocommerce_set_export_product_states' ) ) {
				woocommerce_wp_select(
					array(
						'id'      => 'empik_product_state',
						'label'   => esc_html__( 'Empik product state', 'empik-for-woocommerce' ),
						'options' => $product_states,
						'value'   => get_post_meta( $post->ID, '_empik_product_state', true ),
					)
				);
			}
		}
		?>
		<script>
			(function ($) {

				$('.EmpikHasDatepicker').each(function (ind, elem) {
					$(elem).datepicker({
						dateFormat: 'yy-mm-dd', // ISO formatting date
					});
				});

				$('.empik_price_reset').on('click', function (e) {
					e.preventDefault();
					$('input[id*=empik_sale_date]').each(function (ind, elem) {
						$(elem).val('');
					});
					$('#empik_price_sale').val('');
				});

				$('#empik_price_sale').on("input", function () {

					let empik_regular_price = compare_empik_inputs();

					let empik_sale_price = +$(this).val();

					if (empik_sale_price >= empik_regular_price) {
						$(this).after(
							'<div class="wc_error_tip">Cena promocyjna powinna być niższa od ceny standardowej</div>'
						);
						$('.EmpikHasDatepicker').each(function (ind, elem) {
							$(elem).prop('required', false);
						});
					} else {

						if (empik_sale_price > 0) {
							$('.EmpikHasDatepicker').each(function (ind, elem) {
								$(elem).prop('required', true);
							});

						} else {

							$('.EmpikHasDatepicker').each(function (ind, elem) {
								$(elem).prop('required', false);
							});
						}
					}

				});

				$('#empik_price_sale').on("focusout", function () {

					let empik_regular_price = compare_empik_inputs();

					if (+$(this).val() >= empik_regular_price) {
						$('#empik_price_sale').val('');
					}
				});

				function compare_empik_inputs() {
					let empik_regular_price = $('#empik_price').val();
					if (typeof empik_regular_price != 'undefined' && empik_regular_price !== null) {
						return +empik_regular_price;
					}
					return 0;
				}
			})(jQuery);
		</script>

		<?php
	}


	/**
	 * Displays Empik pricing settings fields for product variations.
	 *
	 * Renders custom price fields specific to product variations including regular
	 * price, sale price, sale date range, and product state selection. Includes
	 * JavaScript validation for sale price logic and date picker functionality
	 * with variation-specific field naming.
	 *
	 * @param int    $loop           The variation loop index.
	 * @param array  $variation_data The variation data array.
	 * @param object $variation      The variation post object.
	 */
	public function empik_product_price_settings_variation( $loop, $variation_data, $variation ) {

		global $post;

		$price_value   = get_post_meta( $variation->ID, '_empik_price', true );
		$price_regular = get_post_meta( $variation->ID, '_regular_price', true );

		if ( ! get_option( 'empik_woocommerce_set_export_offers_from' ) ) {
			$price_percent_multiplier = (float) Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_OFFERS_PRICE_MULTIPLIER );

			$price_add_value = (float) Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_OFFERS_ADD_TO_PRICE );

			if ( $price_percent_multiplier > 0 ) {
				$price_value = (float) $price_regular + ( (float) $price_regular * ( $price_percent_multiplier / 100 ) );
			}

			if ( $price_add_value > 0 ) {
				$price_value = (float) $price_regular + (float) $price_add_value;
			}
		}

		?>
		<p class="form-field variable_regular_empik_price_<?php echo esc_attr( $loop ); ?>_field form-row form-row-first">
			<label for="variable_regular_price_0">
				<?php echo esc_html__( 'Price Empik', 'empik-for-woocommerce' ) . ' (' . get_woocommerce_currency_symbol() . ')'; ?>
			</label>
			<input type="text" class="short wc_input_price" style=""
					name="empik_price[<?php echo esc_attr( $loop ); ?>]"
					id="empik_price_<?php echo esc_attr( $loop ); ?>"
					value="<?php echo esc_attr( $price_value ); ?>"
					placeholder="">
		</p>


		<p class="form-field variable_regular_empik_price_sale_<?php echo esc_attr( $loop ); ?>_field form-row form-row-last">
			<label for="variable_regular_price_0">
				<?php echo esc_html__( 'Sale price Empik', 'empik-for-woocommerce' ) . ' (' . get_woocommerce_currency_symbol() . ')'; ?>
			</label>
			<input type="text" class="short wc_input_price" style=""
					name="empik_price_sale[<?php echo esc_attr( $loop ); ?>]"
					id="empik_price_sale_<?php echo esc_attr( $loop ); ?>"
					value="<?php echo esc_attr( get_post_meta( $variation->ID, '_empik_price_sale', true ) ); ?>"
					placeholder="">
		</p>

		<p class="form-field variable_empik_sale_date_start_<?php echo esc_attr( $loop ); ?>_field inited inited_media_selector">
			<label for="empik_sale_date_start">
				<?php echo esc_html__( 'Daty promocji, od', 'empik-for-woocommerce' ); ?>
			</label>
			<input type="text" class="short EmpikHasDatepicker" style=""
					name="empik_sale_date_start[<?php echo esc_attr( $loop ); ?>]"
					id="empik_sale_date_start_<?php echo esc_attr( $loop ); ?>"
					placeholder="<?php echo esc_html__( 'From… YYYY-MM-DD', 'empik-for-woocommerce' ); ?>"
					maxlength="10"
					pattern="[0-9]{4}-(0[1-9]|1[012])-(0[1-9]|1[0-9]|2[0-9]|3[01])"
					value="<?php echo esc_attr( get_post_meta( $variation->ID, '_empik_sale_date_start', true ) ); ?>"
			>
		</p>
		<p class="form-field variable_empik_sale_date_end_<?php echo esc_attr( $loop ); ?>_field inited inited_media_selector">
			<label for="empik_sale_date_end">
				<?php echo esc_html__( 'Daty promocji, do', 'empik-for-woocommerce' ); ?>
			</label>
			<input type="text" class="short EmpikHasDatepicker" style=""
					name="empik_sale_date_end[<?php echo esc_attr( $loop ); ?>]"
					id="empik_sale_date_end_<?php echo esc_attr( $loop ); ?>"
					placeholder="<?php echo esc_html__( 'To… YYYY-MM-DD', 'empik-for-woocommerce' ); ?>"
					maxlength="10"
					pattern="[0-9]{4}-(0[1-9]|1[012])-(0[1-9]|1[0-9]|2[0-9]|3[01])"
					value="<?php echo esc_attr( get_post_meta( $variation->ID, '_empik_sale_date_end', true ) ); ?>"
			><br><br>
			<span class="description">
			<a href="#" class="empik_price_reset" data-id="<?php echo esc_attr( $loop ); ?>">
				<?php echo esc_html__( 'Reset Empik settings', 'empik-for-woocommerce' ); ?>
			</a>
		</span><br>
		</p>

		<?php
		$api_factory    = new Api_Factory();
		$api            = $api_factory->create_service();
		$product_states = $api->of61_get_product_states();

		if ( ! empty( $product_states ) && is_array( $product_states ) ) {
			if ( 'yes' === get_option( 'empik_woocommerce_set_export_product_states' ) ) {
				$first_tem      = array( 'same' => esc_html__( 'Same as parent', 'empik-for-woocommerce' ) );
				$product_states = $first_tem + $product_states;

				woocommerce_wp_select(
					array(
						'id'            => 'empik_product_state[' . esc_attr( $loop ) . ']',
						'label'         => esc_html__( 'Empik product state', 'empik-for-woocommerce' ),
						'options'       => $product_states,
						'value'         => esc_attr( get_post_meta( $variation->ID, '_empik_product_state', true ) ),
						'class'         => '',
						'style'         => '',
						'wrapper_class' => 'form-row form-row-full variable_empik_state form-field variable_empik_state_' . esc_attr( $loop ) . '_field',
					)
				);
			}
		}
		?>
		<script>
			(function ($) {

				$('.EmpikHasDatepicker').each(function (ind, elem) {
					$(elem).datepicker({
						dateFormat: 'yy-mm-dd', // ISO formatting date.
					});
				});

				$('.empik_price_reset').on('click', function (e) {
					e.preventDefault();
					let variation_id = $(this).attr('data-id');

					$('input[id=empik_sale_date_end_' + variation_id + ']').val('');
					$('input[id=empik_sale_date_start_' + variation_id + ']').val('');
					$('input[id=empik_price_sale_' + variation_id + ']').val('');

					$('.save-variation-changes').prop('disabled', false);
					$(this).closest('.woocommerce_variation').addClass('variation-needs-update');

				});

				$('input[id=empik_price_sale_<?php echo esc_attr( $loop ); ?>]').on("input", function () {

					let empik_regular_price = compare_empik_inputs(<?php echo esc_attr( $loop ); ?>);

					let empik_sale_price = +$(this).val();

					if (empik_sale_price >= empik_regular_price) {
						$(this).after(
							'<div class="wc_error_tip empik_tip_error">Cena promocyjna powinna być niższa od ceny standardowej</div>'
						);
						document.getElementById("empik_sale_date_start_<?php echo esc_attr( $loop ); ?>").required = false;
						document.getElementById("empik_sale_date_end_<?php echo esc_attr( $loop ); ?>").required = false;
					} else {
						$('.empik_tip_error').remove();
						if (empik_sale_price > 0) {
							document.getElementById("empik_sale_date_start_<?php echo esc_attr( $loop ); ?>").required = true;
							document.getElementById("empik_sale_date_end_<?php echo esc_attr( $loop ); ?>").required = true;

						} else {
							document.getElementById("empik_sale_date_start_<?php echo esc_attr( $loop ); ?>").required = false;
							document.getElementById("empik_sale_date_end_<?php echo esc_attr( $loop ); ?>").required = false;
						}
					}

				});

				$('input[id=empik_price_sale_<?php echo esc_attr( $loop ); ?>]').on("focusout", function () {

					let empik_regular_price = compare_empik_inputs(<?php echo esc_attr( $loop ); ?>);

					if (+$(this).val() >= empik_regular_price) {
						$('#empik_price_sale_<?php echo esc_attr( $loop ); ?>').val('');
					}
				});

				function compare_empik_inputs(variation_ind) {
					let empik_regular_price = $('#empik_price_' + variation_ind).val();
					if (typeof empik_regular_price != 'undefined' && empik_regular_price !== null) {
						return +empik_regular_price;
					}
					return 0;
				}

			})(jQuery);
		</script>

		<?php
	}


	/**
	 * Saves Empik-specific metadata for product variations.
	 *
	 * Processes and saves custom Empik fields for individual product variations
	 * including product state, pricing information, and sale date ranges. Uses
	 * loop index to identify the correct variation data from POST array.
	 *
	 * @param int $variation_id The variation post ID.
	 * @param int $loop         The variation loop index.
	 */
	function empik_save_variation_meta( $variation_id, $loop ) {

		if ( isset( $_POST['empik_product_state'][ $loop ] ) && ! empty( $_POST['empik_product_state'][ $loop ] ) ) {
			update_post_meta( $variation_id, '_empik_product_state', sanitize_text_field( $_POST['empik_product_state'][ $loop ] ) );
		}

		if ( isset( $_POST['empik_price'][ $loop ] ) && ! empty( $_POST['empik_price'][ $loop ] ) ) {
			update_post_meta( $variation_id, '_empik_price', sanitize_text_field( $_POST['empik_price'][ $loop ] ) );
		} else {
			update_post_meta( $variation_id, '_empik_price', '' );
		}

		if ( isset( $_POST['empik_price_sale'][ $loop ] ) && ! empty( $_POST['empik_price_sale'][ $loop ] ) ) {
			update_post_meta( $variation_id, '_empik_price_sale', sanitize_text_field( $_POST['empik_price_sale'][ $loop ] ) );
		} else {
			update_post_meta( $variation_id, '_empik_price_sale', '' );
		}

		if ( isset( $_POST['empik_sale_date_start'][ $loop ] ) && ! empty( $_POST['empik_sale_date_start'][ $loop ] ) ) {
			update_post_meta( $variation_id, '_empik_sale_date_start', sanitize_text_field( $_POST['empik_sale_date_start'][ $loop ] ) );

		} else {
			update_post_meta( $variation_id, '_empik_sale_date_start', '' );
		}

		if ( isset( $_POST['empik_sale_date_end'][ $loop ] ) && ! empty( $_POST['empik_sale_date_end'][ $loop ] ) ) {
			update_post_meta( $variation_id, '_empik_sale_date_end', sanitize_text_field( $_POST['empik_sale_date_end'][ $loop ] ) );

		} else {
			update_post_meta( $variation_id, '_empik_sale_date_end', '' );
		}
	}


	/**
	 * Displays product state selection field for variable products.
	 *
	 * Renders a dropdown field that allows setting the Empik product state for
	 * all variants of a variable product. The field is conditionally shown only
	 * for variable product types and includes JavaScript to toggle visibility
	 * based on product type selection.
	 */
	public function product_states_variable() {

		global $post;

		$api_factory    = new Api_Factory();
		$api            = $api_factory->create_service();
		$product_states = $api->of61_get_product_states();

		if ( ! empty( $product_states ) && is_array( $product_states ) ) {
			if ( 'yes' === get_option( 'empik_woocommerce_set_export_product_states' ) ) {

				$first_item     = array( '0' => esc_html__( 'None', 'woocommerce' ) );
				$product_states = $first_item + $product_states;
				echo "<div id='empik_product_state_variable_product' class='empik_product_custom_field'>";
				woocommerce_wp_select(
					array(
						'id'      => 'empik_product_state_all_variants',
						'label'   => esc_html__( 'Empik product state for all variants', 'empik-for-woocommerce' ),
						'options' => $product_states,
						'value'   => get_post_meta( $post->ID, '_empik_product_state_all_variants', true ),
					)
				);
				echo "</div><script>
                                (function($) {
                                    $(document).ready(function() {
                                        let product_type = $('#product-type').val();
                                        if ( product_type === 'variable' ) {
                                            $( '#empik_product_state_variable_product' ).show();
                                        } else {
                                            $( '#empik_product_state_variable_product' ).hide();
                                        }                                
                                    });
                                    $( 'body' ).on( 'woocommerce-product-type-change', function( event, select_val, select ) {
                                        if ( 'variable' === select_val ) {
                                            $( '#empik_product_state_variable_product' ).show();
                                        } else {
                                            $( '#empik_product_state_variable_product' ).hide();
                                        }
                                    });            
                                })(jQuery);
                            </script>";
			}
		}
	}
}
