<?php
/**
 * Plugin main class.
 *
 * @package Inspire_Labs\Empik_Woocommerce
 */

namespace Inspire_Labs\Empik_Woocommerce;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use EmpikWoocommerceVendor\WPDesk\PluginBuilder\Plugin\Deactivateable;
use Exception;
use Inspire_Labs\Empik_Woocommerce\Api_Server\Ping_Export_Offers_Factory;
use Inspire_Labs\Empik_Woocommerce\Api_Server\Ping_Export_Products_Factory;
use Inspire_Labs\Empik_Woocommerce\Api_Server\Ping_Import_Orders_Factory;
use Inspire_Labs\Empik_Woocommerce\Offer\Offer_Actions_Factory;
use Inspire_Labs\Empik_Woocommerce\Order\Order_Details_Factory;
use Inspire_Labs\Empik_Woocommerce\Order\Order_Importer_Factory;
use Inspire_Labs\Empik_Woocommerce\Product\Product_Settings;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Alerts;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Connection_Test_Factory;
use Inspire_Labs\Empik_Woocommerce\Offer\Offers_Exporter_Factory;
use Inspire_Labs\Empik_Woocommerce\Order\Order_Importer_Factory as Orders_Importer_Factory;
use Inspire_Labs\Empik_Woocommerce\Order\Status_Factory;
use Inspire_Labs\Empik_Woocommerce\Product\Product_Exporter_Factory;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\CSV\CSV_Settings;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Products_List_Mod_Factory;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Ids;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Static_Factory;
use Inspire_Labs\Empik_Woocommerce\Utils\Settings_Helper;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;


use EmpikWoocommerceVendor\WPDesk\PluginBuilder\Plugin\AbstractPlugin;
use EmpikWoocommerceVendor\WPDesk\PluginBuilder\Plugin\HookableCollection;
use EmpikWoocommerceVendor\WPDesk\PluginBuilder\Plugin\HookableParent;
use EmpikWoocommerceVendor\WPDesk_Plugin_Info;
use function False\true;



/**
 * Main plugin class. The most important flow decisions are made here.
 *
 * @package WPDesk\PluginTemplate
 */
class Plugin extends AbstractPlugin implements LoggerAwareInterface, HookableCollection, Deactivateable {
	use LoggerAwareTrait;
	use HookableParent;

	const TEXTDOMAIN = 'empik-for-woocommerce';
	const APP_PREFIX = 'empik_woocommerce';

	public static $plugin_dir;

	/**
	 * Plugin constructor.
	 *
	 * @param WPDesk_Plugin_Info $plugin_info Plugin info.
	 */
	public function __construct( $plugin_info ) {
		parent::__construct( $plugin_info );

		$this->plugin_url       = $this->plugin_info->get_plugin_url();
		$this->plugin_namespace = $this->plugin_info->get_text_domain();

		self::$plugin_dir = $this->get_plugin_file_path();
	}

	/**
	 * Initializes plugin external state.
	 *
	 * The plugin internal state is initialized in the constructor and the plugin should be internally consistent after creation.
	 * The external state includes hooks execution, communication with other plugins, integration with WC etc.
	 *
	 * @return void
	 */
	public function init() {
		parent::init();

		$this->customize_wpdesk_boilerplate();

		require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
		require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';

		if ( is_admin() || wp_doing_ajax() ) {
			$this->init_admin_features();
		}

		$this->receive_cron_jobs();
	}


	/**
	 * Customizes WPDesk boilerplate functionality.
	 *
	 * Removes the WPDesk helper menu page from the admin menu and disables
	 * plugin activation notices to provide a cleaner admin interface experience.
	 */
	private function customize_wpdesk_boilerplate() {
		add_action(
			'admin_menu',
			function () {
				remove_menu_page( 'wpdesk-helper' );
			},
			100
		);

		add_filter(
			'wpdesk_show_plugin_activation_notice',
			function ( $bool ) {
				return false;
			}
		);
	}

	/**
	 * Initializes plugin hooks and cron schedules.
	 *
	 * Sets up parent hooks, admin hooks, and registers cron job callbacks for
	 * importing orders, offers, and products. Conditionally schedules or clears
	 * cron events based on the WP cron enabled setting with staggered delays
	 * to prevent simultaneous execution.
	 */
	public function hooks() {
		parent::hooks();
		$this->admin_hooks();

		add_action( 'empik_wp_cron_import_orders', array( $this, 'empik_wp_cron_import_orders_callback' ), 10 );
		add_action( 'empik_wp_cron_import_offers', array( $this, 'empik_wp_cron_import_offers_callback' ), 10 );
		add_action( 'empik_wp_cron_import_products', array( $this, 'empik_wp_cron_import_products_callback' ), 10 );

		$is_wp_cron_enabled = get_option( 'empik_woocommerce_set_cron_type' );

		if ( '1' === $is_wp_cron_enabled ) {

			add_filter( 'cron_schedules', array( $this, 'add_ten_minute_cron_interval' ) );

			$args  = array();
			$delay = 60;
			if ( ! wp_next_scheduled( 'empik_wp_cron_import_orders', $args ) ) {
				wp_schedule_event( time() + $delay, 'empik_orders_cron', 'empik_wp_cron_import_orders' );
			}
			if ( ! wp_next_scheduled( 'empik_wp_cron_import_offers', $args ) ) {
				wp_schedule_event( time() + $delay + 60, 'hourly', 'empik_wp_cron_import_offers' );
			}
			if ( ! wp_next_scheduled( 'empik_wp_cron_import_products', $args ) ) {
				wp_schedule_event( time() + $delay + 120, 'empik_products_cron', 'empik_wp_cron_import_products' );
			}
		} else {
			wp_clear_scheduled_hook( 'empik_wp_cron_import_orders' );
			wp_clear_scheduled_hook( 'empik_wp_cron_import_offers' );
			wp_clear_scheduled_hook( 'empik_wp_cron_import_products' );
		}
	}


	/**
	 * Initializes cron job receivers for external triggers.
	 *
	 * Creates and initializes ping services for handling external cron job
	 * triggers for exporting offers, exporting products, and importing orders.
	 */
	private function receive_cron_jobs() {
		( new Ping_Export_Offers_Factory() )->create_service()->init();
		( new Ping_Export_Products_Factory() )->create_service()->init();
		( new Ping_Import_Orders_Factory() )->create_service()->init();
	}

	/**
	 * Registers admin-specific hooks.
	 *
	 * Adds the plugin's settings page to WooCommerce settings by filtering
	 * the settings pages array and including the custom settings service.
	 */
	private function admin_hooks() {
		add_filter(
			'woocommerce_get_settings_pages',
			function ( $settings ) {
				$settings[] = Settings_Static_Factory::create_service();

				return $settings;
			},
			100
		);
	}


	/**
	 * Registers frontend-specific hooks.
	 */
	private function frontend_hooks() {
	}


	/**
	 * Initializes all admin-specific features and services.
	 *
	 * Sets up plugin action links, sidebar icons, and initializes various
	 * factories and services including product list modifications, exporters,
	 * importers, order status handling, connection testing, and CSV functionality.
	 *
	 * @return void
	 */
	private function init_admin_features() {
		$this->add_plugin_action_links();
		$this->add_main_sidebar_icon();
		$products_List_Mod_Factory = new Products_List_Mod_Factory();
		$products_List_Mod         = $products_List_Mod_Factory->create_service();
		$products_List_Mod->init();
		$products_exporter_factory = new Product_Exporter_Factory();
		$offers_exporter_factory   = new Offers_Exporter_Factory();
		$products_exporter         = $products_exporter_factory->create_service();
		$products_exporter->init();
		$offers_exporter = $offers_exporter_factory->create_service();
		$offers_exporter->init();

		$orders_importer_factory = new Orders_Importer_Factory();
		$orders_importer         = $orders_importer_factory->create_service();

		$orders_importer->init();
		$order_status_factory = new Status_Factory();
		$order_status_factory->create_service()->init();
		$connection_test_factory = new Connection_Test_Factory();
		$connection_test_factory->create_service()->init();
		( new Order_Details_Factory() )->create_service()->init();
		( new Offer_Actions_Factory() )->create_service()->init();

		( new Wp_Admin\CSV\CSV_Ajax() );
		$product_settings = new Product_Settings();
		$product_settings->init();
	}


	/**
	 * Adds custom action links to the plugin's entry on the WordPress plugins page.
	 *
	 * Removes any existing WPDesk links and adds support and plugin webpage links
	 * that open in new tabs.
	 *
	 * @return void
	 */
	private function add_plugin_action_links() {
		add_filter(
			'plugin_action_links_' . EMPIK_WOOCOMMERCE_PLUGIN_BASENAME,
			function ( $links ) {
				foreach ( $links as $k => $link ) {
					if ( strpos( $link, 'wpdesk' ) !== false ) {
						unset( $links[ $k ] );
						break;
					}
				}

				$links[] = '<a target="_blank" href="https://www.empik.com/marketplace/support/kontakt">' . esc_html__(
					'Support',
					'empik-for-woocommerce'
				) . '</a>';
				$links[] = '<a target="_blank" href="https://www.empik.com/zostan-sprzedawca">' . esc_html__(
					'Visit plugin webpage',
					'empik-for-woocommerce'
				) . '</a>';

				return $links;
			}
		);
	}

	/**
	 * Gets the admin script ID used for enqueuing JavaScript files.
	 *
	 * @return string The admin script identifier with app prefix.
	 */
	private function get_admin_script_id(): string {
		return self::APP_PREFIX . '_admin-js';
	}


	/**
	 * Gets the admin CSS ID used for enqueuing stylesheet files.
	 *
	 * @return string The admin CSS identifier with app prefix.
	 */
	private function get_admin_css_id(): string {
		return self::APP_PREFIX . '_admin-css';
	}


	/**
	 * Enqueues admin scripts and styles for the plugin.
	 *
	 * Registers and enqueues JavaScript files with localized data, conditionally
	 * loads CSS files based on current admin screen and URL parameters, and
	 * handles CSV-specific assets for the main plugin page.
	 *
	 * @return void
	 */
	public function admin_enqueue_scripts() {

		$current_screen = get_current_screen();

		$admin_js_path = EMPIK_WOOCOMMERCE_PLUGIN_PATH . 'assets/js/admin.js';
		$admin_js_ver  = file_exists( $admin_js_path ) ? filemtime( $admin_js_path ) : EMPIK_WOOCOMMERCE_PLUGIN_VERSION;
		wp_register_script(
			$this->get_admin_script_id(),
			$this->get_plugin_assets_url() . '/js/admin.js',
			array( 'jquery' ),
			$admin_js_ver,
			true
		);

		wp_localize_script(
			$this->get_admin_script_id(),
			self::APP_PREFIX,
			array(
				'admin_url'           => admin_url( 'admin-ajax.php' ),
				'connection_btn_text' => esc_html__( 'Test connection', 'empik-for-woocommerce' ),
				'klasa_title'         => esc_html__( 'Send logistic class', 'empik-for-woocommerce' ),
				'state_title'         => esc_html__( 'Send product states', 'empik-for-woocommerce' ),
				'price_notice'        => esc_html__( "Attention! If this option selected, complete the 'Empik Price' field in the product settings", 'empik-for-woocommerce' ),
				'nonce'               => wp_create_nonce( 'empik_woocommerce_products_export' ),
			)
		);

		wp_enqueue_script( $this->get_admin_script_id() );

		if ( is_a( $current_screen, 'WP_Screen' ) && 'woocommerce_page_wc-settings' === $current_screen->id ) {
			if ( isset( $_GET['tab'] ) && sanitize_text_field( $_GET['tab'] ) === 'empik_woocommerce' ) {

				$admin_css_path = EMPIK_WOOCOMMERCE_PLUGIN_PATH . 'assets/css/admin.css';
				wp_enqueue_style(
					$this->get_admin_css_id(),
					$this->get_plugin_assets_url() . '/css/admin.css',
					array(),
					file_exists( $admin_css_path ) ? filemtime( $admin_css_path ) : EMPIK_WOOCOMMERCE_PLUGIN_VERSION
				);

				if ( isset( $_GET['section'] ) && sanitize_text_field( $_GET['section'] ) === 'offers' ) {
					$admin_offers_css_path = EMPIK_WOOCOMMERCE_PLUGIN_PATH . 'assets/css/admin-offers.css';
					wp_enqueue_style(
						'empik-admin-offers',
						$this->get_plugin_assets_url() . '/css/admin-offers.css',
						array(),
						file_exists( $admin_offers_css_path ) ? filemtime( $admin_offers_css_path ) : EMPIK_WOOCOMMERCE_PLUGIN_VERSION
					);
				}
			}
		}

		if ( isset( $_GET['post_type'] ) && sanitize_text_field( $_GET['post_type'] ) === 'product' ) {
			wp_enqueue_style(
				$this->get_admin_css_id(),
				$this->get_plugin_assets_url() . '/css/admin.css'
			);
		}

		if ( is_a( $current_screen, 'WP_Screen' ) && 'toplevel_page_ilabs_empik' === $current_screen->id ) {
			wp_register_script(
				'empik-csv-js',
				$this->get_plugin_assets_url() . '/js/csv-script.js',
				array( 'jquery' ),
				EMPIK_WOOCOMMERCE_PLUGIN_VERSION,
				true
			);

			wp_localize_script(
				'empik-csv-js',
				'empik_csv',
				array(
					'ajaxurl'   => admin_url( 'admin-ajax.php' ),
					'postnonce' => wp_create_nonce( 'empik_csv_nonce' ),
				)
			);

			wp_enqueue_script( 'empik-csv-js' );

			if ( isset( $_GET['steps'] ) && sanitize_text_field( $_GET['steps'] ) === 'table' ) {
				wp_enqueue_style(
					'empik-csv-css',
					$this->get_plugin_assets_url() . '/css/csv_import_styles.css'
				);
			}
		}
	}

	/**
	 * Gets a new instance of the Settings Helper.
	 *
	 * @return Settings_Helper A new settings helper instance.
	 */
	public static function get_settings(): Settings_Helper {
		return new Settings_Helper();
	}

	/**
	 * Custom log
	 *
	 * @param $log_msg $log_msg.
	 */
	public static function log( $log_msg ) {
	}

	/**
	 * Writes debug information to the error log.
	 *
	 * Handles arrays and objects by converting them to readable format,
	 * while logging scalar values directly.
	 *
	 * @param mixed $log The data to log (string, array, object, etc.).
	 * @return void
	 */
	public static function write_debug_log( $log ) {
		if ( is_array( $log ) || is_object( $log ) ) {
			error_log( print_r( $log, true ) );
		} else {
			error_log( $log );
		}
	}

	/**
	 * On deactivate
	 */
	public function deactivate() {
		update_option( self::APP_PREFIX . '_is_registered', false );
		wp_clear_scheduled_hook( 'empik_wp_cron_import_orders' );
		wp_clear_scheduled_hook( 'empik_wp_cron_import_offers' );
		wp_clear_scheduled_hook( 'empik_wp_cron_import_products' );
	}

	/**
	 * Add admin menu icon
	 */
	private function add_main_sidebar_icon() {
		add_action( 'admin_menu', array( $this, 'admin_menu_items' ) );
	}


	/**
	 * Add admin menu items
	 */
	public function admin_menu_items() {
		global $menu;
		$menu_pos = 56;
		while ( isset( $menu[ $menu_pos ] ) ) {
			++$menu_pos;
		}

		// Icon.
		$icon = 'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBzdGFuZGFsb25lPSJubyI/Pgo8IURPQ1RZUEUgc3ZnIFBVQkxJQyAiLS8vVzNDLy9EVEQgU1ZHIDIwMDEwOTA0Ly9FTiIKICJodHRwOi8vd3d3LnczLm9yZy9UUi8yMDAxL1JFQy1TVkctMjAwMTA5MDQvRFREL3N2ZzEwLmR0ZCI+CjxzdmcgdmVyc2lvbj0iMS4wIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciCiB3aWR0aD0iNDguMDAwMDAwcHQiIGhlaWdodD0iNDguMDAwMDAwcHQiIHZpZXdCb3g9IjAgMCA0OC4wMDAwMDAgNDguMDAwMDAwIgogcHJlc2VydmVBc3BlY3RSYXRpbz0ieE1pZFlNaWQgbWVldCI+Cgo8ZyB0cmFuc2Zvcm09InRyYW5zbGF0ZSgwLjAwMDAwMCw0OC4wMDAwMDApIHNjYWxlKDAuMTAwMDAwLC0wLjEwMDAwMCkiCmZpbGw9IiMwMDAwMDAiIHN0cm9rZT0ibm9uZSI+CjxwYXRoIGQ9Ik0wIDI0MCBsMCAtMjQwIDI0MCAwIDI0MCAwIDAgMjQwIDAgMjQwIC0yNDAgMCAtMjQwIDAgMCAtMjQweiBtMzE3CjMyIGMtNSAtODMgLTI3IC0xNDEgLTY2IC0xNzYgLTIxIC0yMCAtMjIgLTIwIC00MSAwIC0yMCAyMCAtMjAgMjAgNSA1NCAxNCAxOQoyNSAzOCAyNSA0MiAwIDQgLTE4IDggLTQwIDggbC00MCAwIDAgNzUgMCA3NSA4MSAwIDgyIDAgLTYgLTc4eiIvPgo8L2c+Cjwvc3ZnPgo=';

		// Add menu icon.
		add_menu_page(
			esc_html__( 'Empik', 'empik-for-woocommerce' ),
			esc_html__( 'Empik', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'ilabs_empik',
			null,
			$icon,
			$menu_pos
		);

		add_submenu_page(
			'ilabs_empik',
			esc_html__( 'Connection', 'empik-for-woocommerce' ),
			esc_html__( 'Connection', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'admin.php?page=wc-settings&tab=empik_woocommerce',
			null
		);

		add_submenu_page(
			'ilabs_empik',
			esc_html__( 'Products', 'empik-for-woocommerce' ),
			esc_html__( 'Products', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'admin.php?page=wc-settings&tab=empik_woocommerce&section=products',
			null
		);

		add_submenu_page(
			'ilabs_empik',
			esc_html__( 'Offers', 'empik-for-woocommerce' ),
			esc_html__( 'Offers', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'admin.php?page=wc-settings&tab=empik_woocommerce&section=offers',
			null
		);

		add_submenu_page(
			'ilabs_empik',
			esc_html__( 'Orders', 'empik-for-woocommerce' ),
			esc_html__( 'Orders', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'admin.php?page=wc-settings&tab=empik_woocommerce&section=orders',
			null
		);

		add_submenu_page(
			'ilabs_empik',
			esc_html__( 'CRON actions', 'empik-for-woocommerce' ),
			esc_html__( 'CRON actions', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'admin.php?page=wc-settings&tab=empik_woocommerce&section=cron-actions',
			null
		);

		add_submenu_page(
			'',
			esc_html__( 'Bulk change of logistics class', 'empik-for-woocommerce' ),
			esc_html__( 'Bulk change of logistics class', 'empik-for-woocommerce' ),
			'view_woocommerce_reports',
			'ilabs_empik',
			array( $this, 'render_csv_page' )
		);

		remove_submenu_page( 'ilabs_empik', 'ilabs_empik' );
	}

	/**
	 * Render settings page for CSV import
	 */
	public function render_csv_page() {

		new CSV_Settings();
	}


	/**
	 * Adds custom cron intervals for Empik plugin operations.
	 *
	 * Registers a 10-minute interval for order synchronization and an 8-hour
	 * interval for product synchronization with Empik marketplace.
	 *
	 * @param array $schedules Existing WordPress cron schedules.
	 * @return array Modified schedules array with new intervals.
	 */
	public function add_ten_minute_cron_interval() {

		$schedules['empik_orders_cron'] = array(
			'interval' => 600, // 10 minutes in seconds (10 * 60).
			'display'  => 'Every 10 Minutes (Empik Orders)',
		);

		$schedules['empik_products_cron'] = array(
			'interval' => 28800, // 8 hours in seconds (8 * 60 * 60).
			'display'  => 'Every 8 Hours (Empik Products)',
		);

		return $schedules;
	}


	/**
	 * WordPress cron callback for importing orders from Empik.
	 *
	 * Checks for valid API key and prevents concurrent execution before
	 * triggering the order import process via the order importer service.
	 *
	 * @return void
	 */
	public function empik_wp_cron_import_orders_callback() {

		$api_key = get_option( 'empik_woocommerce_set_api_key' );

		if ( empty( $api_key ) ) {
			return;
		}

		if ( ! empty( get_option( 'empik_import_orders_is_running' ) ) ) {
			return;
		}

		$orders_importer = ( new Order_Importer_Factory() )->create_service();
		// WP CRON, import orders.
		$result = $orders_importer->import();

		//$local_time = time() + 7200;
		//$when       = gmdate( 'H:i:s', $local_time );
		//\wc_get_logger()->debug( 'WP CRON, import orders: ' . $when, array( 'source' => 'empik-wp-cron' ) );
		//\wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-wp-cron' ) );
	}


	/**
	 * WordPress cron callback for exporting offers to Empik.
	 *
	 * Checks for valid API key and prevents concurrent execution before
	 * triggering the offer export process via the offers exporter service.
	 *
	 * @return void
	 */
	public function empik_wp_cron_import_offers_callback() {

		$api_key = get_option( 'empik_woocommerce_set_api_key' );

		if ( empty( $api_key ) ) {
			return;
		}

		if ( ! empty( get_option( 'empik_export_offers_is_running' ) ) ) {
			return;
		}

		$offers_exporter_factory = new Offers_Exporter_Factory();
		$offers_exporter         = $offers_exporter_factory->create_service();
		// WP CRON, offers export to API.
		$result = $offers_exporter->export();

		//$local_time = time() + 7200;
		//$when       = gmdate( 'H:i:s', $local_time );
		//\wc_get_logger()->debug( 'WP CRON, offers export to API: ' . $when, array( 'source' => 'empik-wp-cron' ) );
		//\wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-wp-cron' ) );
	}


	/**
	 * WordPress cron callback for exporting products to Empik.
	 *
	 * Checks for valid API key and prevents concurrent execution before
	 * triggering the product export process via the products exporter service.
	 *
	 * @return void
	 */
	public function empik_wp_cron_import_products_callback() {

		$api_key = get_option( 'empik_woocommerce_set_api_key' );

		if ( empty( $api_key ) ) {
			return;
		}

		if ( ! empty( get_option( 'empik_export_products_is_running' ) ) ) {
			return;
		}

		$products_exporter_factory = new Product_Exporter_Factory();
		$products_exporter         = $products_exporter_factory->create_service();
		// WP CRON, products export to API.
		$result = $products_exporter->export();

		//$local_time = time() + 7200;
		//$when       = gmdate( 'H:i:s', $local_time );
		//\wc_get_logger()->debug( 'WP CRON, products export to API: ' . $when, array( 'source' => 'empik-wp-cron' ) );
		//\wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-wp-cron' ) );
	}
}
