<?php

namespace Inspire_Labs\Empik_Woocommerce\Order;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Exception;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Abstract_Api;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Ajax_Response_Model;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Api_Factory;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Testing_Api_Factory;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Ids;
use stdClass;
use WC_Checkout;
use WC_Order;
use WC_Product;
use WC_Shipping_Rate;
use WP_Query;

/**
 * Order_Importer
 */
class Order_Importer {
	const EMPIK_ORDER_ID_META_KEY     = '_empik_order_id';
	const EMPIK_ORDER_OBJECT_META_KEY = '_empik_order_object';


	/**
	 * Get API additional parameters.
	 *
	 * @return array
	 */
	protected function get_api_additional_parameters(): array {
		return array();
	}


	/**
	 * Get context phrase
	 *
	 * @return string
	 */
	protected function get_context_phrase(): string {
		return 'orders';
	}

	/**
	 * Add Ajax callback
	 *
	 * @return void
	 */
	public function init() {
		add_action( $this->get_ajax_action_name(), array( $this, 'ajax_callback' ) );
	}


	/**
	 * Get Ajax action name
	 *
	 * @return string
	 */
	protected function get_ajax_action_name(): string {
		return 'wp_ajax_' . $this->get_module_id();
	}

	/**
	 * Get module id
	 *
	 * @return string
	 */
	private function get_module_id(): string {
		$context_phrase = $this->get_context_phrase();

		return Plugin::APP_PREFIX . '_' . $context_phrase . '_import';
	}

	/**
	 * Ajax callback
	 *
	 * @return void
	 */
	public function ajax_callback() {

		$result = $this->import();

		if ( $result ) {
			wp_send_json( $result->to_array(), 200 );
		}

		exit;
	}


	/**
	 *  Should auto confirm orders
	 *
	 * @return bool
	 */
	private function should_auto_confirm(): bool {
		$option = Plugin::get_settings()->get_option( ( Settings_Ids::SETTINGS_ID_CONFIRM_ORDERS_AUTOMATICALLY ) );
		if ( '0' === $option ) {
			return false;
		}
		return true;
	}

	/**
	 * Should auto download orders
	 *
	 * @return bool
	 */
	private function should_auto_download(): bool {
		$option = Plugin::get_settings()->get_option( ( Settings_Ids::SETTINGS_ID_DOWNLOAD_ORDERS_AUTOMATICALLY ) );
		if ( '0' === $option ) {
			return false;
		}
		return true;
	}

	/**
	 * Import
	 *
	 * @return Ajax_Response_Model
	 * @throws Exception Exception.
	 */
	public function import(): Ajax_Response_Model {

		$debug = '<hr>Nowe żądanie. ' . gmdate( 'Y-m-d H:i:s' ) . '<br>';

		if ( $this->should_auto_confirm() ) {
			$debug .= $this->confirm();
		}

		$debug .= '<br><br><br>';

		if ( $this->should_auto_download() ) {
			$debug .= $this->download();
		}
		Plugin::log( $debug );

		return ( new Ajax_Response_Model(
			Ajax_Response_Model::SUCCESS,
			$debug
		)
		);
	}

	/**
	 * Confirm orders
	 *
	 * @return string
	 * @throws Exception Exception.
	 */
	private function confirm(): string {

		update_option( 'empik_import_orders_is_running', true, true );

		$debug_text = '<h3>Rozpoczęto zadanie AKCEPTACJI zamówień</h3><br>';
		$wc_debug   = 'Nowe żądanie. ' . date( 'Y-m-d H:i:s' );
		$wc_debug  .= PHP_EOL . 'Rozpoczęto zadanie AKCEPTACJI zamówień' . PHP_EOL;

		try {
			$api_factory = new Api_Factory();
			$api         = $api_factory->create_service();
			$result      = $api->import( Abstract_Api::ORDER_STATUS_WAITING_ACCEPTANCE );

			if ( ! $result || ! property_exists( $result, 'orders' ) ) {
				\wc_get_logger()->debug( 'Import zamówień Exception: ', array( 'source' => 'empik-import-orders-log' ) );
				\wc_get_logger()->debug( print_r( $result->message, true ), array( 'source' => 'empik-import-orders-log' ) );

				delete_option( 'empik_import_orders_is_running' );

				return $result->message;
			}

			$debug_text .= 'URL zapytania: /api/orders?order_state_codes=WAITING_ACCEPTANCE<br>';
			$debug_text .= 'API zwróciło łącznie zamówień: ' . count( $result->orders ) . '<br>';
			$wc_debug   .= 'API zwróciło łącznie zamówień: ' . count( $result->orders ) . PHP_EOL;

			if ( count( $result->orders ) === 0 ) {
				$debug_text .= 'Brak zamówień do przetworzenia<br>';
				$wc_debug   .= 'Brak zamówień do przetworzenia' . PHP_EOL;

				delete_option( 'empik_import_orders_is_running' );
			}

			foreach ( $result->orders as $empik_order ) {
				$debug_text .= '<br><br>Przetwarzanie zamówienia: ' . $empik_order->order_id . '<br>';
				$wc_debug   .= PHP_EOL;
				$wc_debug   .= PHP_EOL . 'Przetwarzanie zamówienia: ' . $empik_order->order_id . PHP_EOL;

				$debug_text .= 'Liczba produktów w zamówieniu: ' . count( $empik_order->order_lines ) . '<br>';
				$wc_debug   .= 'Liczba produktów w zamówieniu: ' . count( $empik_order->order_lines ) . PHP_EOL;

				$products_valid = true;
				$order_lines    = array();

				if ( empty( $empik_order->order_lines ) ) {
					$debug_text .= 'Nie znaleziono produktów w zamówieniu<br>';
					$wc_debug   .= 'Nie znaleziono produktów w zamówieniu' . PHP_EOL;
				}
				foreach ( $empik_order->order_lines as $empik_order_line ) {
					if ( 0 === $wc_product_id = wc_get_product_id_by_sku(
						$empik_order_line->offer_sku
					) ) {
						$products_valid = false;
						$debug_text    .= 'Nie znaleziono w bazie danych produktu o SKU: ' . $empik_order_line->offer_sku . '<br>';
						$wc_debug      .= 'Nie znaleziono w bazie danych produktu o SKU: ' . $empik_order_line->offer_sku . PHP_EOL;
						break;
					}
					$woo_product    = wc_get_product( $wc_product_id );
					$empik_quantity = $empik_order_line->quantity;
					$woo_in_stock   = $woo_product->is_in_stock();
					$woo_quantity   = $woo_product->get_stock_quantity();
					if ( ! $woo_in_stock || $woo_quantity < $empik_quantity ) {
						$products_valid = false;
						$debug_text    .= 'Znaleziono produkt, ale nie ma go w magazynie<br>';
						$wc_debug      .= 'Znaleziono produkt, ale nie ma go w magazynie' . PHP_EOL;
						break;
					}

					$order_lines[] = array(
						'accepted' => 'true',
						'id'       => $empik_order_line->order_line_id,
					);
				}

				if ( ! $products_valid ) {
					$debug_text .= 'Odrzucono zamówienie ' . $empik_order->order_id . ' z powodu niedopasowania produktów.<br>';
					$wc_debug   .= 'Odrzucono zamówienie ' . $empik_order->order_id . ' z powodu niedopasowania produktów.' . PHP_EOL;
					continue;
				} else {
					$debug_text .= 'Zweryfikowano wszystkie produkty<br>';
					$wc_debug   .= 'Zweryfikowano wszystkie produkty' . PHP_EOL;
				}

				$this->accept_order_api( $empik_order->order_id, $order_lines );

				$debug_text .= 'Zaakceptowano zamówienie ' . $empik_order->order_id . ' metodą OR-21<br>';
				$wc_debug   .= 'Zaakceptowano zamówienie ' . $empik_order->order_id . ' metodą OR-21' . PHP_EOL;
			}

			$debug_text .= 'Akceptowanie zamówień ukończone<br>';
			$wc_debug   .= 'Akceptowanie zamówień ukończone' . PHP_EOL;

			\wc_get_logger()->debug( 'Import zamówień: ', array( 'source' => 'empik-import-orders-log' ) );
			\wc_get_logger()->debug( print_r( $wc_debug, true ), array( 'source' => 'empik-import-orders-log' ) );

			return $debug_text;

		} catch ( Exception $e ) {
			\wc_get_logger()->debug( 'ERROR: Import zamówień: ', array( 'source' => 'empik-import-orders-log' ) );
			\wc_get_logger()->debug( print_r( $e->getMessage(), true ), array( 'source' => 'empik-import-orders-log' ) );
			\wc_get_logger()->debug( print_r( $result->message, true ), array( 'source' => 'empik-import-orders-log' ) );

			delete_option( 'empik_import_orders_is_running' );

			return $e->getMessage();
		}
	}

	/**
	 * Download orders
	 *
	 * @return string
	 * @throws Exception Exception.
	 */
	private function download(): string {

		$debug_text = '<h3>Rozpoczęto zadanie POBIERANIA zamówień</h3><br>';
		$wc_debug   = 'Rozpoczęto zadanie POBIERANIA zamówień' . PHP_EOL;
		try {
			$api_factory = new Api_Factory();
			$api         = $api_factory->create_service();
			$result      = $api->import( Abstract_Api::ORDER_STATUS_SHIPPING );

			if ( ! $result || ! property_exists( $result, 'orders' ) ) {
				\wc_get_logger()->debug( 'ERROR: POBIERANIA zamówień: ', array( 'source' => 'empik-import-orders-log' ) );
				\wc_get_logger()->debug( print_r( 'NO property orders exists', true ), array( 'source' => 'empik-import-orders-log' ) );
				\wc_get_logger()->debug( print_r( $result->message, true ), array( 'source' => 'empik-import-orders-log' ) );

				delete_option( 'empik_import_orders_is_running' );

				return $result->message;
			}

			$debug_text .= 'URL zapytania: /api/orders?order_state_codes=SHIPPING<br>';
			$debug_text .= 'API zwróciło łącznie zamówień: ' . count( $result->orders ) . '<br><br>';
			$wc_debug   .= 'API zwróciło łącznie zamówień: ' . count( $result->orders ) . PHP_EOL;

			foreach ( $result->orders as $empik_order ) {

				$debug_text .= '<br><br>Przetwarzanie zamówienia: ' . $empik_order->order_id . '<br>';
				$wc_debug   .= PHP_EOL;
				$wc_debug   .= PHP_EOL . 'Przetwarzanie zamówienia: ' . $empik_order->order_id . PHP_EOL;

				if ( $this->is_order_exists_by_empik_id( $empik_order->order_id ) ) {
					$debug_text .= 'Zamówienie o tym ID istnieje już w bazie danych. Pomiń.</br>';
					$wc_debug   .= 'Zamówienie o tym ID istnieje już w bazie danych. Pomiń.' . PHP_EOL;
					continue;
				}

				$order = wc_create_order();

				$debug_text .= 'Zainicjowano zamówienie Woocommerce. Empik ID: ' . $empik_order->order_id . '<br>';
				$wc_debug   .= 'Zainicjowano zamówienie Woocommerce. Empik ID: ' . $empik_order->order_id . PHP_EOL;

				$address_shipping = array(
					'first_name' => $this->get_from_std( $empik_order->customer->shipping_address, 'firstname' ),
					'last_name'  => $this->get_from_std( $empik_order->customer->shipping_address, 'lastname' ),
					'company'    => $this->get_from_std( $empik_order->customer->shipping_address, 'company' ),
					'phone'      => $this->get_from_std( $empik_order->customer->shipping_address, 'phone' ),
					'address_1'  => $this->get_from_std( $empik_order->customer->shipping_address, 'street_1' ),
					'address_2'  => $this->get_from_std( $empik_order->customer->shipping_address, 'street_2' ),
					'city'       => $this->get_from_std( $empik_order->customer->shipping_address, 'city' ),
					'postcode'   => $this->get_from_std( $empik_order->customer->shipping_address, 'zip_code' ),
					'country'    => $this->get_from_std( $empik_order->customer->shipping_address, 'country_iso_code' ),
				);

				$address_billing = array(
					'first_name' => $this->get_from_std( $empik_order->customer->billing_address, 'firstname' ),
					'last_name'  => $this->get_from_std( $empik_order->customer->billing_address, 'lastname' ),
					'company'    => $this->get_from_std( $empik_order->customer->billing_address, 'company' ),
					'phone'      => $this->get_from_std( $empik_order->customer->billing_address, 'phone' ),
					'address_1'  => $this->get_from_std( $empik_order->customer->billing_address, 'street_1' ),
					'address_2'  => $this->get_from_std( $empik_order->customer->billing_address, 'street_2' ),
					'city'       => $this->get_from_std( $empik_order->customer->billing_address, 'city' ),
					'postcode'   => $this->get_from_std( $empik_order->customer->billing_address, 'zip_code' ),
					'country'    => $this->get_from_std( $empik_order->customer->billing_address, 'country_iso_code' ),
					'email'      => $this->get_email_from_empik_order( $empik_order ),
				);

				$order->set_address( $address_billing, 'billing' );
				$order->set_address( $address_shipping, 'shipping' );

				$debug_text .= 'Liczba produktów w zamówieniu: ' . count( $empik_order->order_lines ) . '<br>';
				$wc_debug   .= 'Liczba produktów w zamówieniu: ' . count( $empik_order->order_lines ) . PHP_EOL;

				$products_valid = true;
				if ( empty( $empik_order->order_lines ) ) {
					$debug_text .= 'Nie znaleziono produktów w zamówieniu<br>';
					$wc_debug   .= 'Nie znaleziono produktów w zamówieniu' . PHP_EOL;
				}

				$is_shipping_included_tax = false;

				foreach ( $empik_order->order_lines as $empik_order_line ) {

					$is_shipping_included_tax = false;

					if ( 0 === $wc_product_id = wc_get_product_id_by_sku(
						$empik_order_line->offer_sku
					) ) {
						$products_valid = false;
						$debug_text    .= 'Nie znaleziono w bazie danych produktu o SKU: ' . $empik_order_line->offer_sku . '<br>';
						$wc_debug      .= 'Nie znaleziono w bazie danych produktu o SKU: ' . $empik_order_line->offer_sku . PHP_EOL;
						break;
					}

					$woo_product    = wc_get_product( $wc_product_id );
					$empik_quantity = $empik_order_line->quantity;
					$empik_price    = $empik_order_line->price_unit;

					$woo_in_stock = $woo_product->is_in_stock();
					$woo_quantity = $woo_product->get_stock_quantity();
					if ( ! $woo_in_stock || $woo_quantity < $empik_quantity ) {
						$products_valid = false;
						$debug_text    .= 'Znaleziono produkt, ale nie ma go w magazynie<br>';
						$wc_debug      .= 'Znaleziono produkt, ale nie ma go w magazynie' . PHP_EOL;
						break;
					}

					$item_rate = 0;

					if ( 'yes' === get_option( 'woocommerce_calc_taxes' ) ) {

						$empik_price = floatval( $empik_price );

						if ( $woo_product->is_taxable() ) {

							$tax = new \WC_Tax();

							if ( is_object( $tax ) ) {

								$taxes = $tax->get_rates( $woo_product->get_tax_class() );

								if ( ! empty( $taxes ) ) {

									$rates = array_shift( $taxes );

									if ( ! empty( $rates ) ) {
										// Take only the item rate and round it.
										$item_rate = round( array_shift( $rates ) );

										// is tax calculated for shipping costs.
										if ( isset( $rates['shipping'] ) && 'yes' === $rates['shipping'] ) {
											$is_shipping_included_tax = true;
										}
									}

									$tax_amount = $empik_price - ( $empik_price / ( ( $item_rate / 100 ) + 1 ) );

									$wc_debug .= 'Is_taxable: YES' . PHP_EOL;
									$wc_debug .= 'TAX amount: ' . $tax_amount . PHP_EOL;
									$wc_debug .= 'Empik price before: ' . $empik_price . PHP_EOL;

									$price_without_tax = $empik_price - $tax_amount;

									$wc_debug .= 'Empik price without tax: ' . $price_without_tax . PHP_EOL;

									$empik_price = $price_without_tax;

								}
							}
						}
					}

					$order->add_product( wc_get_product( $wc_product_id ), $empik_quantity );
					$debug_text .= 'Dodano produkt: ' . $woo_product->get_name() . '<br>';
					$debug_text .= 'Set_prices_include_tax: ' . get_option( 'woocommerce_prices_include_tax' ) . '<br>';
					$wc_debug   .= 'Dodano produkt: ' . $woo_product->get_name() . PHP_EOL;
					$wc_debug   .= 'Set_prices_include_tax: ' . get_option( 'woocommerce_prices_include_tax' ) . PHP_EOL;

					$order->set_prices_include_tax( 'yes' === get_option( 'woocommerce_prices_include_tax' ) );
					$order->save();

					$this->update_wc_order_line_price_by_empik_price( $order, $woo_product, $empik_price );
				}

				if ( ! $products_valid ) {
					$debug_text .= 'Ignorowanie ' . $empik_order->order_id . ' z powodu niedopasowania produktów.<br>';
					$wc_debug   .= 'Ignorowanie ' . $empik_order->order_id . ' z powodu niedopasowania produktów' . PHP_EOL;
					wp_delete_post( $order->get_id(), true );
					if ( 'yes' === get_option( 'woocommerce_custom_orders_table_enabled' ) ) {
						$order->delete( true );
					}

					continue;

				} else {
					$debug_text .= 'Poprawnie dodano wszystkie produkty do zamówienia<br>';
					$wc_debug   .= 'Poprawnie dodano wszystkie produkty do zamówienia' . PHP_EOL;
				}

				$order->calculate_totals();

				$order->update_meta_data( self::EMPIK_ORDER_ID_META_KEY, $empik_order->order_id );

				$order->update_meta_data( self::EMPIK_ORDER_OBJECT_META_KEY, $empik_order );

				$order->update_meta_data( '_created_via', 'Empik Woocommerce' );

				$order->update_meta_data( '_wc_order_attribution_utm_source', 'Import z Empik' );
				$order->update_meta_data( '_wc_order_attribution_source_type', 'utm' );

				$order->add_order_note(
					esc_html__(
						'Empik Order ID: ',
						'empik-for-woocommerce'
					) . esc_attr( $empik_order->order_id )
				);

				$order->add_order_note(
					esc_html__(
						'Payment type: ',
						'empik-for-woocommerce'
					) . esc_html( $this->get_payment_type_from_empik_order( $empik_order ) )
				);

				$payment_workflow = $this->get_payment_workflow_from_empik_order( $empik_order );

				if ( 'PAY_ON_ACCEPTANCE' === $payment_workflow ) {
					$order->update_meta_data( '_date_paid', time() );
				}

				if ( property_exists( $empik_order, 'shipping_price' ) ) {
					$order->update_meta_data( '_order_shipping', $empik_order->shipping_price );
				}

				if ( 'PAY_ON_DELIVERY' === $payment_workflow ) {
					$order->update_meta_data( '_payment_method', 'cod' );
					$order->update_meta_data( '_date_paid', null );
					$order->update_meta_data( '_paid_date', null );
				}

				if ( ! empty( $empik_order->shipping_price ) ) {

					$empik_shipping_cost   = $empik_order->shipping_price;
					$empik_shipping_method = esc_html__( 'From Empik Order', 'empik-for-woocommerce' );
					if ( isset( $empik_order->shipping_type_label ) && ! empty( $empik_order->shipping_type_label ) ) {
						$empik_shipping_method = $empik_order->shipping_type_label;
					}

					if ( $is_shipping_included_tax ) {

						$empik_shipping_cost = floatval( $empik_shipping_cost );

						$shipping_tax_amount        = $empik_shipping_cost - ( $empik_shipping_cost / ( ( $item_rate / 100 ) + 1 ) );
						$shipping_price_without_tax = $empik_shipping_cost - $shipping_tax_amount;
						$empik_shipping_cost        = $shipping_price_without_tax;

					}

					$order->set_shipping_total( $empik_shipping_cost );

					$shipping_object = new \WC_Order_Item_Shipping();

					$shipping_object->set_shipping_rate(
						new WC_Shipping_Rate(
							'empik_woocommerce',
							$empik_shipping_method,
							$empik_shipping_cost
						)
					);

					$order->add_item(
						$shipping_object
					);

					$order->calculate_shipping();
					$order->calculate_totals();

					if ( 0 !== $this->verify_order_total( $order, $empik_order->total_price ) ) {
						$correct_amount_after_round = $this->verify_order_total( $order, $empik_order->total_price );

						$empik_shipping_cost_corrected = $empik_shipping_cost - $correct_amount_after_round;

						$items = (array) $order->get_items( 'shipping' );
						if ( count( $items ) > 0 ) {
							foreach ( $items as $item_id => $item ) {
								$order->remove_item( $item_id );
							}
							$order->calculate_totals();
						}

						$order->set_shipping_total( $empik_shipping_cost_corrected );
						$shipping_object = new \WC_Order_Item_Shipping();

						$shipping_object->set_shipping_rate(
							new WC_Shipping_Rate(
								'empik_woocommerce',
								$empik_shipping_method,
								$empik_shipping_cost_corrected,
								array()
							)
						);
						$order->add_item(
							$shipping_object
						);
						$order->calculate_shipping();
						$order->calculate_totals();
						$order->set_total( $empik_order->total_price );

					}
				}

				$order->add_order_note(
					esc_html__(
						'Shipping type: ',
						'empik-for-woocommerce'
					) . esc_html( $this->get_shipping_type_from_empik_order( $empik_order ) )
				);

				$order->add_order_note(
					esc_html__( 'NIP: ', 'empik-for-woocommerce' ) . esc_attr( $this->get_nip_from_empik_order( $empik_order, $order ) )
				);

				$order->update_status( 'processing', 'Źródło zamówienia: Empik', true );

				$debug_text .= 'Koniec procesowania zamówienia Empik ID: ' . $empik_order->order_id . '<br>';
				$wc_debug   .= 'Koniec procesowania zamówienia Empik ID: ' . $empik_order->order_id . PHP_EOL;

				$res_order_id = $order->save();

				$debug_text .= 'Utworzono zamówienie pod numerem: ' . $res_order_id . '<br>';
				$wc_debug   .= 'Utworzono zamówienie pod numerem: ' . $res_order_id . PHP_EOL;
			}

			$debug_text .= 'Pobieranie ukońcozne<br>';
			$wc_debug   .= 'Pobieranie ukońcozne' . PHP_EOL;

			\wc_get_logger()->debug( 'Pobieranie zamówień ukońcozne: ', array( 'source' => 'empik-import-orders-log' ) );
			\wc_get_logger()->debug( print_r( $wc_debug, true ), array( 'source' => 'empik-import-orders-log' ) );

			delete_option( 'empik_import_orders_is_running' );

			return $debug_text;

		} catch ( Exception $e ) {
			Plugin::write_debug_log( $e->getMessage() );
			\wc_get_logger()->debug( 'ERROR: Pobieranie zamówień: ', array( 'source' => 'empik-import-orders-log' ) );
			\wc_get_logger()->debug( print_r( $e->getMessage(), true ), array( 'source' => 'empik-import-orders-log' ) );
			\wc_get_logger()->debug( print_r( $result->message, true ), array( 'source' => 'empik-import-orders-log' ) );

			delete_option( 'empik_import_orders_is_running' );

			return $e->getMessage();
		}
	}


	/**
	 * Update WC_Order line price by empik price
	 *
	 * @param WC_Order    $order $order.
	 * @param WC_Product  $wc_product $wc_product.
	 * @param $empik_price $empik_price.
	 * @return void
	 */
	private function update_wc_order_line_price_by_empik_price(
		WC_Order $order,
		WC_Product $wc_product,
		$empik_price
	) {

		$product_parent_id = $wc_product->get_parent_id();
		$pid               = '';
		if ( 0 === $product_parent_id ) {
			$pid = $wc_product->get_id();
		} else {
			$pid = $product_parent_id;
		}

		/**
		 * WC_Order_Item_Product $item
		 *
		 * @var \WC_Order_Item_Product $item
		 */
		foreach ( $order->get_items() as $item_id => $item ) {
			if ( $item->get_product_id() === $pid ) {
				$product_quantity    = (int) $item->get_quantity();
				$new_line_item_price = $empik_price * $product_quantity;
				$item->set_subtotal( $new_line_item_price );
				$item->set_total( $new_line_item_price );

				$item->save();
				$order->calculate_totals();
				break;
			}
		}
	}

	/**
	 * Get payment type from empik order
	 *
	 * @param stdClass $order $order.
	 *
	 * @return mixed|string|void
	 */
	private function get_payment_type_from_empik_order( stdClass $order ) {
		return property_exists( $order, 'paymentType' )
			? $order->paymentType
			: esc_html__( 'Payment type not found in order', 'empik-for-woocommerce' );
	}

	/**
	 * Get payment workflow from empik order
	 *
	 * @param stdClass $order $order.
	 *
	 * @return mixed|string|void
	 */
	private function get_payment_workflow_from_empik_order( stdClass $order ) {
		return property_exists( $order, 'payment_workflow' )
			? $order->payment_workflow
			: null;
	}

	/**
	 * Get shipping type from empik order
	 *
	 * @param stdClass $order $order.
	 *
	 * @return mixed|string|void
	 */
	private function get_shipping_type_from_empik_order( stdClass $order ) {
		return property_exists( $order, 'shipping_type_label' )
			? $order->shipping_type_label
			: esc_html__( 'Shipping type not found in order', 'empik-for-woocommerce' );
	}

	/**
	 * Get NIP from empik order
	 *
	 * @param stdClass $order $order.
	 * @param WC_Order $wc_order $wc_order.
	 * @return mixed|string|void
	 */
	private function get_nip_from_empik_order( stdClass $order, WC_Order $wc_order ) {

		if ( is_array( $order->order_additional_fields ) ) {
			foreach ( $order->order_additional_fields as $additional_field ) {
				if (
					is_object( $additional_field )
					&& property_exists( $additional_field, 'code' )
					&& 'nip' === $additional_field->code
				) {
					$wc_order->update_meta_data( '_billing_vat_number', $additional_field->value );
					$wc_order->update_meta_data( 'nip', $additional_field->value );
					return $additional_field->value;
				}
			}
		}

		return esc_html__( 'NIP not found in order', 'empik-for-woocommerce' );
	}

	/**
	 * Get Email from empik order
	 *
	 * @param stdClass $order $order.
	 *
	 * @return mixed|string
	 */
	private function get_email_from_empik_order( stdClass $order ) {

		if ( is_array( $order->order_additional_fields ) ) {
			foreach ( $order->order_additional_fields as $additional_field ) {
				if (
					is_object( $additional_field )
					&& property_exists( $additional_field, 'code' )
					&& 'customer-email' === $additional_field->code
				) {
					return $additional_field->value;
				}
			}
		}

		return esc_html__( 'Email not found in order', 'empik-for-woocommerce' );
	}


	/**
	 * Check is order exists by empik_id
	 *
	 * @param string $empik_order_id $empik_order_id.
	 *
	 * @return bool
	 */
	private function is_order_exists_by_empik_id( string $empik_order_id ) {

		\wc_get_logger()->debug( PHP_EOL . ' Debug function "is_order_exists_by_empik_id": ', array( 'source' => 'empik-debug-log' ) );
		\wc_get_logger()->debug( print_r( $empik_order_id, true ), array( 'source' => 'empik-debug-log' ) );

		$args = array(
			'fields'         => 'ids',
			'post_status'    => 'any',
			'post_type'      => 'shop_order',
			'posts_per_page' => - 1,
			'meta_query'     =>
				array(
					array(
						'key'     => self::EMPIK_ORDER_ID_META_KEY,
						'value'   => $empik_order_id,
						'compare' => '=',
					),
				),
		);

		$result = get_posts( $args );

		if ( empty( $result ) ) {
			// HPOS.

			\wc_get_logger()->debug( print_r( 'empty result', true ), array( 'source' => 'empik-debug-log' ) );

			if ( 'yes' === get_option( 'woocommerce_custom_orders_table_enabled' ) ) {

				\wc_get_logger()->debug( print_r( 'HPOS', true ), array( 'source' => 'empik-debug-log' ) );

				global $wpdb;

				$meta_key   = self::EMPIK_ORDER_ID_META_KEY;
				$meta_value = $empik_order_id;

				$query = $wpdb->prepare(
					"
                        SELECT order_id
                        FROM {$wpdb->prefix}wc_orders_meta
                        WHERE meta_key = %s
                        AND meta_value = %s
                        ",
					$meta_key,
					$meta_value
				);

				$result = $wpdb->get_col( $query );

			}
		}

		\wc_get_logger()->debug( print_r( 'result wpdb: ', true ), array( 'source' => 'empik-debug-log' ) );
		\wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-debug-log' ) );

		if ( is_array( $result ) && ! empty( $result ) && is_numeric( $result[0] ) ) {
			\wc_get_logger()->debug( print_r( PHP_EOL . 'Dla zamówienia ' . $empik_order_id . ' w sklepie istnieje już wcześniejsze zamówienie, nie importujemy!' . PHP_EOL, true ), array( 'source' => 'empik-debug-log' ) );
		}

		return is_array( $result ) && ! empty( $result ) && is_numeric( $result[0] );
	}

	/**
	 * Accept order API
	 *
	 * @param string $order_id $order_id.
	 * @param array  $order_lines $order_lines.
	 *
	 * @return bool
	 */
	private function accept_order_api( string $order_id, array $order_lines ): bool {
		try {
			( new Api_Factory() )->create_service()->or_21_order_accept( $order_id, $order_lines );

			return true;
		} catch ( Exception $e ) {
			return false;
		}
	}

	/**
	 * Get from std
	 *
	 * @param object $object $object.
	 * @param string $property $property.
	 *
	 * @return mixed|null
	 */
	private function get_from_std( $object, $property ) {
		return property_exists( $object, $property )
			? $object->$property
			: null;
	}


	/**
	 * Verifies if order total matches Empik order amount.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param WC_Order $order The WooCommerce order object.
	 * @param float    $empik_order_amount The Empik order amount to verify against.
	 * @return float The difference between calculated total and Empik amount.
	 */
	private function verify_order_total( $order, $empik_order_amount ) {
		$order_subtotal       = $order->get_subtotal();
		$order_total_tax      = $order->get_total_tax();
		$order_shipping_total = $order->get_shipping_total();

		return floatval( $order_subtotal ) + floatval( $order_total_tax ) + floatval( $order_shipping_total ) - floatval( $empik_order_amount );
	}
}
