<?php

namespace Inspire_Labs\Empik_Woocommerce\Order;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


use Inspire_Labs\Empik_Woocommerce\Api_Client\Api_Factory;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Abstract_Ajax;
use WC_Order;

/**
 * Order_Details
 */
class Order_Details extends Abstract_Ajax {

	/**
	 * Create service
	 *
	 * @return Order_Details
	 */
	public function create_service(): Order_Details {
		return new Order_Details();
	}

	/**
	 * Initializes order meta hooks and actions.
	 *
	 * Extends parent initialization and adds WooCommerce order meta display and save hooks.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 */
	public function init(): void {

		parent::init();

		add_action( 'woocommerce_admin_order_data_after_order_details', array( $this, 'empik_order_meta_r23' ) );
		add_action( 'woocommerce_process_shop_order_meta', array( $this, 'save_general_details' ) );
	}


	/**
	 * Displays Empik order meta fields in WooCommerce order details.
	 *
	 * Shows carrier selection, tracking number input and send to Empik button for Empik orders.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param WC_Order $order The WooCommerce order object.
	 * @return void
	 */
	public function empik_order_meta_r23( WC_Order $order ) {

		$empik_order_id = get_post_meta( $order->get_id(), Order_Importer::EMPIK_ORDER_ID_META_KEY, true );

		// HPOS.
		if ( ! $empik_order_id && 'yes' === get_option( 'woocommerce_custom_orders_table_enabled' ) ) {
			$empik_order_id = $order->get_meta( Order_Importer::EMPIK_ORDER_ID_META_KEY );
		}

		if ( ! empty( $empik_order_id ) ) {

			echo '<br class="clear"/>
            <div class="empik_r23_wrap">';
			echo '<h4>' . esc_html__( 'Empik', 'empik-for-woocommerce' ) . ' <a href="#" class="edit_address">' . esc_html__( 'Edit', 'empik-for-woocommerce' ) . '</a></h4>';

			$carrier_code = get_post_meta( $order->get_id(), '_empik_carrier_code', true );
			if ( ! $carrier_code && 'yes' === get_option( 'woocommerce_custom_orders_table_enabled' ) ) {
				$carrier_code = $order->get_meta( '_empik_carrier_code' );
			}

			$carrier_url = get_post_meta( $order->get_id(), '_empik_carrier_url', true );
			if ( ! $carrier_url && 'yes' === get_option( 'woocommerce_custom_orders_table_enabled' ) ) {
				$carrier_url = $order->get_meta( '_empik_carrier_url' );
			}

			$carrier_tracking_number = get_post_meta( $order->get_id(), '_empik_carrier_tracking_number', true );
			if ( ! $carrier_tracking_number && 'yes' === get_option( 'woocommerce_custom_orders_table_enabled' ) ) {
				$carrier_tracking_number = $order->get_meta( '_empik_carrier_tracking_number' );
			}

			echo '<div class="edit_address">';

			$button_id = $this->get_module_id() . '_btn';
			echo '<button id=' . esc_attr( $button_id ) . '>';
			echo esc_html__( 'Send to Empik', 'empik-for-woocommerce' );
			echo '</button>';

			$carriers          = array();
			$carriers_list_get = ( new Carrier_List_Factory() )->create_service()->get_from_cache();
			$carriers_list     = isset( $carriers_list_get['carriers'] ) && is_array( $carriers_list_get['carriers'] )
				? $carriers_list_get['carriers']
				: array();

			if ( is_array( $carriers_list ) ) {
				foreach ( $carriers_list as $carrier ) {
					$carriers[ $carrier->code ] = $carrier->label;
				}

				woocommerce_wp_select(
					array(
						'id'          => $this->get_module_id() . '_code',
						'label'       => esc_html__( 'Carrier', 'empik-for-woocommerce' ),
						'options'     => $carriers,
						'desc_tip'    => true,
						'description' => '',
						'value'       => $carrier_code,
					)
				);

				echo '</div>';
				echo '<div class="edit_address">';

				echo '</div>';
				echo '<div class="edit_address">';

				woocommerce_wp_text_input(
					array(
						'id'            => $this->get_module_id() . '_number',
						'label'         => esc_html__( 'Carrier tracking number', 'empik-for-woocommerce' ),
						'value'         => $carrier_tracking_number,
						'wrapper_class' => 'form-field-wide',
					)
				);

				$hidden_field_id       = $this->get_module_id() . '_order_id';
				$hidden_field_order_id = $empik_order_id;

				echo '<input type="hidden" id ="empik_wc_order_id" value="' . esc_attr( $order->get_id() ) . '">';
				echo '<input type="hidden" value="' . esc_attr( $hidden_field_order_id ) . '" id="' . esc_attr( $hidden_field_id ) . '">';
				wp_nonce_field( 'empik_carrier_update', 'empik_carrier_update' );
				echo '</div>';
				echo '</div>';
			}
		}
	}



	/**
	 * Saves Empik carrier details for an order.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param int $ord_id The order ID to save carrier details for.
	 * @return void
	 */
	public function save_general_details( $ord_id ) {

		if ( ! isset( $_POST['empik_woocommerce_carrier_info_code'] ) ) {
			return;
		}

		update_post_meta(
			$ord_id,
			'_empik_carrier_tracking_number',
			sanitize_text_field( $_POST['empik_woocommerce_carrier_info_number'] )
		);
		update_post_meta(
			$ord_id,
			'_empik_carrier_code',
			sanitize_text_field( $_POST['empik_woocommerce_carrier_info_code'] )
		);
	}


	/**
	 * Handles AJAX request for updating carrier information.
	 *
	 * Validates request data, updates carrier details via API, and returns JSON response.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void Sends JSON response and exits.
	 */
	public function ajax_callback() {

		if ( ! isset( $_POST['order_id'] )
			|| ! isset( $_POST['info_code'] )
			|| ! isset( $_POST['info_number'] )
		) {
			return;
		}

		if ( ! wp_doing_ajax() ) {
			return;
		}

		check_ajax_referer( 'empik_carrier_update', 'security' );

		$api_factory = new Api_Factory();
		$api         = $api_factory->create_service();

		$result = null;
		$code   = null;
		$name   = null;

		if ( ! empty( $_POST['info_code'] ) ) {
			$code = sanitize_text_field( wp_unslash( $_POST['info_code'] ) );
		}

		if ( ! empty( $_POST['info_name'] ) ) {
			$name = sanitize_text_field( wp_unslash( $_POST['info_name'] ) );
		}

		if ( $code && $name ) {
			$result = $api->or_23_carrier_update(
				$this->sanitize_key( sanitize_text_field( wp_unslash( $_POST['order_id'] ) ) ),
				$this->sanitize_key( $code ),
				'',
				$this->sanitize_key( sanitize_text_field( wp_unslash( $_POST['info_number'] ) ) ),
				$this->sanitize_key( $name )
			);
		}

		if ( $result ) {
			wp_send_json( $result->to_array(), 200 );
		} else {
			wp_send_json( 'Some error is occurred', 200 );
		}
	}

	/**
	 * Sanitize key
	 *
	 * @param mixed $key $key.
	 *
	 * @return array|string|string[]|null
	 */
	private function sanitize_key( $key ) {
		return $key;

		// return preg_replace( '/[^a-z0-9_\-]/', '', $key );
	}

	/**
	 * Get context phrase
	 *
	 * @return string
	 */
	protected function get_context_phrase(): string {
		return 'carrier_info';
	}

	/**
	 * Get module id
	 *
	 * @return string
	 */
	private function get_module_id(): string {
		return Plugin::APP_PREFIX . '_' . $this->get_context_phrase();
	}
}
