<?php

namespace Inspire_Labs\Empik_Woocommerce\Offer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Exception;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Ajax_Response_Model;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Abstract_Exporter;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Products_List_Offers_import_Column;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Products_List_Product_import_Column;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Ids;
use WC_Product;
use WC_Product_Variable;

/**
 * Offers_Exporter
 */
class Offers_Exporter extends Abstract_Exporter {

	const MUST_BE_EMPTY = '';
	const STATE         = 11;

	const OFFER_ID_TYPE_EAN = 'EAN';
	const OFFER_ID_TYPE_SKU = 'SHOP_SKU';

	/**
	 * @var array
	 */
	private $offer_must_remove_delist_flag;

	/**
	 * Performs post-export actions based on the Ajax response status.
	 *
	 * If export was successful, removes delist flags from specified product offers.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @param Ajax_Response_Model $ajax_response_model The Ajax response model containing status information.
	 * @return void
	 */
	protected function do_after_export_action(
		Ajax_Response_Model $ajax_response_model
	) {
		if ( $ajax_response_model->get_status()
			=== Ajax_Response_Model::SUCCESS
		) {
			if ( ! empty( $this->offer_must_remove_delist_flag ) ) {
				foreach ( $this->offer_must_remove_delist_flag as $post_id ) {
					( new Products_List_Offers_import_Column() )->remove_waiting_to_delist_flag( $post_id );
				}
			}
		}
	}

	/**
	 * Gets the exporter type identifier.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return int The offers exporter type constant.
	 */
	protected function get_type(): int {
		return Abstract_Exporter::EXPORTER_TYPE_OFFERS;
	}


	/**
	 * Gets the context identifier for offers export.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return string The context phrase for offers export.
	 */
	protected function get_context_phrase(): string {
		return 'offers_export';
	}


	/**
	 * Creates CSV file with product offers data.
	 *
	 * Processes simple and variable products, applies price multipliers and additional values,
	 * handles sale prices and dates, and generates offer data for export.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return string Generated CSV content or error message if SKUs are missing.
	 */
	protected function create_csv(): string {

		ini_set( 'max_execution_time', '28800' );

		if ( isset( $_POST['price_percentage_multiplier'] ) ) {
			$price_percent_multiplier
				= (float) sanitize_text_field( $_POST['price_percentage_multiplier'] );
		} else {
			$price_percent_multiplier = (float) Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_OFFERS_PRICE_MULTIPLIER );
		}

		if ( isset( $_POST['price_add_value'] ) ) {
			$price_add_value = (float) sanitize_text_field( $_POST['price_add_value'] );
		} else {
			$price_add_value = (float) Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_OFFERS_ADD_TO_PRICE );
		}

		$products           = array();
		$products_empty_sku = array();

		$args = array(
			'post_type'      => 'product',
			'posts_per_page' => - 1,
			'fields'         => 'ids',
		);

		$is_all_offers = Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ALL );

		if ( 'yes' === $is_all_offers ) {

			$products = get_posts( $args );

		} else {

			$args = apply_filters(
				'offer_exporter_get_products_args',
				$this->update_args_only_checked_offers_or_delist_waiting( $args )
			);

			$products = get_posts( $args );
		}

		$offers = array();

		$offer_id_type = $this->get_offer_id_type();
		if ( self::OFFER_ID_TYPE_EAN === $offer_id_type ) {
			$offer_ean_meta_key = $this->get_offer_ean_id_meta_key();
		} else {
			$offer_ean_meta_key = '';
		}

		$is_price_from_product_settings = false;

		if ( isset( $_POST['price_from_product'] ) && (int) $_POST['price_from_product'] === 1 ) {
			$is_price_from_product_settings = true;
		}

		if ( get_option( 'empik_woocommerce_set_export_offers_from' ) ) {
			$is_price_from_product_settings = true;
		}

		try {
			$i = 0;
			foreach ( $products as $post_id ) {
				// if ($i >= 1000) break;

				$p = wc_get_product( $post_id );
				if ( 'simple' === $p->get_type() ) {

					$price             = '';
					$sale_price        = '';
					$date_on_sale_from = '';
					$date_on_sale_to   = '';

					if ( $is_price_from_product_settings ) {

						$price = get_post_meta( $post_id, '_empik_price', true )
							? get_post_meta( $post_id, '_empik_price', true )
							: $p->get_regular_price();

						$sale_price = get_post_meta( $post_id, '_empik_price_sale', true );

					} else {

						$price      = $p->get_regular_price();
						$sale_price = get_post_meta( $post_id, '_empik_price_sale', true );
						// $sale_price we get from custom field:
						// redmine #25758
						// attachments/download/24152/Empik_Ceny_Mnozhnik.jpg

						// zwiększenie ceny o procent.
						if ( $price_percent_multiplier > 0 ) {
							$price = (string) ( (float) $price + ( (float) $price * ( $price_percent_multiplier / 100 ) ) );
						} else {
							$price_percent_minus = abs( intval( $price_percent_multiplier ) );
							$price               = (string) ( (float) $price - ( (float) $price * ( $price_percent_minus / 100 ) ) );
						}

						// zwiększenie ceny o wartość.
						if ( $price_add_value > 0 || $price_add_value < 0 ) {
							$price = (string) ( (float) $price + (float) $price_add_value );
						}
					}

					if ( ! empty( $sale_price ) && (float) $sale_price > 0 ) {

						$date_on_sale_from = get_post_meta( $post_id, '_empik_sale_date_start', true );
						$date_on_sale_to   = get_post_meta( $post_id, '_empik_sale_date_end', true );

						$date_on_sale_from = $date_on_sale_from . 'T00:00:00.000+02:00';
						$date_on_sale_to   = $date_on_sale_to . 'T23:59:59.000+02:00';
					}

					$offer = new Offer_Model();

					$update_delete = $this->decide_update_delete( $post_id );

					if ( 'delete' === $update_delete ) {
						$this->offer_must_remove_delist_flag[] = $post_id;
					}

					$offer->set_sku( $p->get_sku() );
					$offer->set_id(
						$this->get_offer_id(
							$offer_id_type,
							$offer_ean_meta_key,
							$p
						)
					);
					$offer->set_id_type( $offer_id_type );
					$offer->set_internal_description( '' );
					$offer->set_price_additional_info( '' );

					$qty_control = get_option( 'empik_woocommerce_set_export_offers_stock_control', 0 );
					$stock_qty   = (int) $p->get_stock_quantity();
					if ( $stock_qty > $qty_control ) {
						$offer->set_quantity( $stock_qty );
					} else {
						$offer->set_quantity( '0' );
					}

					$offer->set_min_quantity_alert( $p->get_min_purchase_quantity() );

					$offer->set_available_start_date( self::MUST_BE_EMPTY );
					$offer->set_available_end_date( '' );
					$offer->set_logistic_class( $this->define_logistic_klass( $post_id ) );
					$offer->set_product_state( $this->define_product_state( $post_id ) );
					$offer->set_update_delete( $update_delete );
					$offer->set_discount_start_date( $date_on_sale_from );
					$offer->set_discount_end_date( $date_on_sale_to );
					$offer->set_price( $price );
					$offer->set_discount_price( $sale_price );
					$offer->set_discount_ranges( array() );
					$offer->set_price_ranges( array() );
					$offer->set_discount_start_date_channel( array() );
					$offer->set_price_channel( array() );
					$offer->set_discount_price_channel( array() );
					$offer->set_discount_ranges_channel( array() );
					$offer->set_prices_ranges_channel( array() );
					$offer->set_leadtime_to_ship( array() );

					$offers[] = $offer;
				}

				// !! variable products.
				if ( $p->get_type() === 'variable' ) {

					$update_delete = $this->decide_update_delete( $post_id );

					if ( 'delete' === $update_delete ) {
						$this->offer_must_remove_delist_flag[] = $post_id;
					}

					$logistic_klass = $this->define_logistic_klass( $p->get_id() );
					$product_state  = $this->define_product_state_variable( $p->get_id() );

					$variations = $this->get_all_variations( $p );

					$offer = new Offer_Model();

					$offer->set_sku( $p->get_sku() );
					$offer->set_id(
						$this->get_offer_id(
							$offer_id_type,
							$offer_ean_meta_key,
							$p
						)
					);
					$offer->set_id_type( $offer_id_type );
					$offer->set_internal_description( '' );
					$offer->set_price_additional_info( '' );
					$offer->set_quantity( '' );
					$offer->set_min_quantity_alert( $p->get_min_purchase_quantity() );

					$offer->set_available_start_date( self::MUST_BE_EMPTY );
					$offer->set_available_end_date( '' );
					$offer->set_logistic_class( $logistic_klass );
					$offer->set_product_state( $product_state );
					$offer->set_update_delete( $update_delete );
					$offer->set_price( '' );
					$offer->set_discount_price( '' );
					$offer->set_discount_ranges( array() );
					$offer->set_price_ranges( array() );
					$offer->set_discount_start_date_channel( array() );
					$offer->set_price_channel( array() );
					$offer->set_discount_price_channel( array() );
					$offer->set_discount_ranges_channel( array() );
					$offer->set_prices_ranges_channel( array() );
					$offer->set_leadtime_to_ship( array() );

					$offers[] = $offer;

					foreach ( $variations as $variant ) {

						$price             = '';
						$sale_price        = '';
						$date_on_sale_from = '';
						$date_on_sale_to   = '';

						if ( empty( $variant['sku'] ) ) {
							// data for error notice if SKU is empty.
							if ( ! in_array( $p->get_id(), $products_empty_sku ) ) {
								$products_empty_sku[ $p->get_id() ] = $p->get_name();
							}
						}

						if ( $is_price_from_product_settings ) {

							$price = get_post_meta( $variant['variation_id'], '_empik_price', true )
								? get_post_meta( $variant['variation_id'], '_empik_price', true )
								: $variant['display_regular_price'];

							$sale_price = get_post_meta( $variant['variation_id'], '_empik_price_sale', true );

							if ( ! empty( $sale_price ) && (float) $sale_price > 0 ) {

								$date_on_sale_from = get_post_meta( $variant['variation_id'], '_empik_sale_date_start', true );
								$date_on_sale_to   = get_post_meta( $variant['variation_id'], '_empik_sale_date_end', true );

								$date_on_sale_from = $date_on_sale_from . 'T00:00:00.000+02:00';
								$date_on_sale_to   = $date_on_sale_to . 'T23:59:59.000+02:00';
							}
						} else {

							$price      = $variant['display_regular_price'];
							$sale_price = get_post_meta( $variant['variation_id'], '_empik_price_sale', true );

							// zwiększenie ceny o procent.
							if ( $price_percent_multiplier > 0 ) {
								$price = (string) ( (float) $price + ( (float) $price * ( $price_percent_multiplier / 100 ) ) );
							} else {
								$price_percent_minus = abs( intval( $price_percent_multiplier ) );
								$price               = (string) ( (float) $price - ( (float) $price * ( $price_percent_minus / 100 ) ) );
							}

							// zwiększenie ceny o wartość.
							if ( $price_add_value > 0 || $price_add_value < 0 ) {
								$price = (string) ( (float) $price + (float) $price_add_value );
							}

							if ( ! empty( $sale_price ) && (float) $sale_price > 0 ) {

								$date_on_sale_from = get_post_meta( $variant['variation_id'], '_empik_sale_date_start', true );
								$date_on_sale_to   = get_post_meta( $variant['variation_id'], '_empik_sale_date_end', true );

								$date_on_sale_from = $date_on_sale_from . 'T00:00:00.000+02:00';
								$date_on_sale_to   = $date_on_sale_to . 'T23:59:59.000+02:00';
							}
						}

						$offer = new Offer_Model();

						$offer->set_sku( $variant['sku'] );

						$product_state = $this->define_product_state_variable( $p->get_id(), $variant['variation_id'] );

						$offer->set_id(
							$this->get_offer_id_variant(
								$offer_id_type,
								$offer_ean_meta_key,
								$variant
							)
						);

						$offer->set_id_type( $offer_id_type );

						$qty_control = get_option( 'empik_woocommerce_set_export_offers_stock_control', 0 );
						$stock_qty   = (int) $variant['max_qty'];
						if ( $stock_qty > $qty_control ) {
							$offer->set_quantity( $stock_qty );
						} else {
							$offer->set_quantity( '0' );
						}
						// $offer->set_quantity( (int) $variant['max_qty'] );

						$offer->set_min_quantity_alert( $p->get_min_purchase_quantity() );

						$offer->set_price( $price );
						$offer->set_discount_price( $sale_price );
						$offer->set_logistic_class( $logistic_klass );
						$offer->set_product_state( $product_state );
						$offer->set_discount_start_date( $date_on_sale_from );
						$offer->set_discount_end_date( $date_on_sale_to );
						$offer->set_update_delete( $update_delete );

						$offers[] = $offer;

					}
				}

				++$i;
			}
		} catch ( Exception $e ) {
			\wc_get_logger()->debug( 'Błąd eksportu ofert:', array( 'source' => 'empik-log' ) );
			\wc_get_logger()->debug( print_r( $e->getMessage(), true ), array( 'source' => 'empik-log' ) );
		}

		$csv_factory = new Offer_Csv_Factory();
		$csv         = $csv_factory->create_service();

		/*
		if( ! empty( $products_empty_sku ) ) {
			$error_string_msg = '';
			foreach ($products_empty_sku as $product_id => $product_name) {
				$error_string_msg .= $product_id . ' - ' . $product_name . '<br>' . PHP_EOL;
			}

			return $error_string_msg;
		}
		*/

		return $csv->create_csv( $offers );
	}

	/**
	 * Determines if a product offer should be updated or deleted.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param int $post_id The post ID to check delist status for.
	 * @return string Returns 'delete' if product is marked for delisting, empty string otherwise.
	 */
	private function decide_update_delete( $post_id ): string {

		return ( new Products_List_Offers_import_Column() )->get_meta_value_true()
		=== get_post_meta(
			$post_id,
			( new Products_List_Offers_import_Column() )->get_delist_offer_meta_id(),
			true
		)
			? 'delete'
			: '';
	}

	/**
	 * Gets the offer ID based on specified type and product.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string     $offer_id_type The type of offer ID to retrieve.
	 * @param string     $offer_ean_meta_key The meta key for EAN number.
	 * @param WC_Product $product The product object to get ID for.
	 * @return string The offer ID value.
	 */
	private function get_offer_id(
		string $offer_id_type,
		string $offer_ean_meta_key,
		WC_Product $product
	): string {
		if ( self::OFFER_ID_TYPE_EAN === $offer_id_type ) {
			if ( is_array(
				$meta = get_post_meta(
					$product->get_id(),
					$offer_ean_meta_key
				)
			)
				&& isset( $meta[0] )
			) {
				return $meta[0];
			} else {
				return '';
			}
		}

		return $product->get_sku();
	}

	/**
	 * Gets the offer ID for a product variant.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $offer_id_type The type of offer ID to retrieve.
	 * @param string $offer_ean_meta_key The meta key for EAN number.
	 * @param array  $variant The variant data array.
	 * @return string The variant offer ID value.
	 */
	private function get_offer_id_variant(
		string $offer_id_type,
		string $offer_ean_meta_key,
		$variant
	): string {
		if ( self::OFFER_ID_TYPE_EAN === $offer_id_type ) {
			if ( is_array(
				$meta = get_post_meta(
					$variant['variation_id'],
					$offer_ean_meta_key
				)
			)
				&& isset( $meta[0] )
			) {
				return $meta[0];
			} else {
				return '';
			}
		}
		return $variant['sku'];
	}


	/**
	 * Gets the configured offer ID type from plugin settings.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return string Returns EAN or SKU type constant based on settings.
	 */
	private function get_offer_id_type(): string {
		return Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ID_TYPE )
			=== 'ean'
				? self::OFFER_ID_TYPE_EAN
				: self::OFFER_ID_TYPE_SKU;
	}

	/**
	 * Gets the meta key for EAN ID from plugin settings.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return string The configured EAN meta field key.
	 */
	private function get_offer_ean_id_meta_key(): string {
		return Plugin::get_settings()
				->get_option( Settings_Ids::SETTINGS_ID_EXPORT_OFFERS_ID_FIELD );
	}


	/**
	 * Updates query arguments to filter only checked offers or those waiting for delisting.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @param array $args The original query arguments.
	 * @return array Modified query arguments with meta query conditions.
	 */
	protected function update_args_only_checked_offers_or_delist_waiting(
		array $args
	): array {
		$args['meta_query']['relation'] = 'OR';
		$args['meta_query'][]           = array(
			'key'     => ( new Products_List_Offers_import_Column() )->get_export_offer_meta_id(),
			'value'   => ( new Products_List_Offers_import_Column() )->get_meta_value_true(),
			'compare' => '=',
		);

		$args['meta_query'][] = array(
			'key'     => ( new Products_List_Offers_import_Column() )->get_delist_offer_meta_id(),
			'value'   => ( new Products_List_Offers_import_Column() )->get_meta_value_true(),
			'compare' => '=',
		);

		return $args;
	}

	/**
	 * Processes the export request with the generated CSV file.
	 *
	 * Handles test mode, empty SKU validation, and API export calls.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @param string $path_to_csv Path to the generated CSV file.
	 * @return Ajax_Response_Model Response with success/error status and message.
	 */
	protected function do_request( string $path_to_csv ): Ajax_Response_Model {

		update_option( 'empik_export_offers_is_running', true, true );

		if ( defined( 'EMPIK_TEST_MODE_ON' ) && true === EMPIK_TEST_MODE_ON ) {

			// for testing - generate .csv file but do no export do Empik dashboard.
			if ( ! empty( get_option( 'empik_last_export_file_name' ) ) ) {

				$uploads               = wp_upload_dir();
				$last_export_file_url  = esc_url( $uploads['baseurl'] . '/' . get_option( 'empik_last_export_file_name' ) );
				$last_export_file_name = sanitize_text_field( get_option( 'empik_last_export_file_name' ) );

				delete_option( 'empik_last_export_file_name' );

				delete_option( 'empik_export_offers_is_running' );

				return ( new Ajax_Response_Model(
					Ajax_Response_Model::SUCCESS,
					sprintf(
						"%s %s <br>%s <a download href='%s'>%s</a>",
						esc_html__( 'Export finished. ID:', 'empik-for-woocommerce' ),
						'TEST offers export',
						esc_html__( 'File:', 'empik-for-woocommerce' ),
						$last_export_file_url,
						$last_export_file_name
					)
				) );
			} else {

				delete_option( 'empik_export_offers_is_running' );

				return ( new Ajax_Response_Model(
					Ajax_Response_Model::SUCCESS,
					sprintf(
						esc_html__( 'Export finished. ID: %s ', 'empik-for-woocommerce' ),
						'TEST offers export'
					)
				) );
			}
		} else {

			if ( ! stripos( $path_to_csv, '.csv' ) ) {

				delete_option( 'empik_export_offers_is_running' );

				// we got list of products with empty SKU in variants.
				return ( new Ajax_Response_Model(
					Ajax_Response_Model::ERROR,
					sprintf(
						esc_html__( 'Export failed. Found products with empty SKU in variants (ID - Name):', 'empik-for-woocommerce' ) . '<br>%s',
						$path_to_csv
					)
				) );
			}

			$result = $this->api->export_offers( $path_to_csv );

			delete_option( 'empik_export_offers_is_running' );

			return $result;

		}
	}

	/**
	 * Determines the logistic class for a product.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param int $pid The product ID to check.
	 * @return string Returns logistic class value ('0', custom value, or default '2').
	 */
	private function define_logistic_klass( $pid ) {
		$logistic_klass = get_post_meta( $pid, '_empik_logistic_klass', true );

		if ( '0' === $logistic_klass ) {
			return '0';
		} elseif ( ! empty( $logistic_klass ) ) {
			return $logistic_klass;
		} else {
			// default value = 2.
			return '2';
		}
	}


	/**
	 * Determines the product state value.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param int $pid The product ID to check.
	 * @return string Returns custom product state if set, or default '11' (New).
	 */
	private function define_product_state( $pid ) {
		$product_state = get_post_meta( $pid, '_empik_product_state', true );

		if ( ! empty( $product_state ) && is_numeric( $product_state ) ) {
			return $product_state;
		} else {
			// default value = 11 Nowy.
			return '11';
		}
	}


	/**
	 * Determines the product state for variable products and their variations.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param int      $product_id The parent product ID.
	 * @param int|null $variation_id Optional variation ID to check.
	 * @return string The product state value based on variation and parent settings.
	 */
	private function define_product_state_variable( $product_id, $variation_id = null ) {
		$product_state = '11';
		if ( $variation_id ) {
			$variation_state = get_post_meta( $variation_id, '_empik_product_state', true );

			if ( ! empty( $variation_state ) ) {

				if ( is_numeric( $variation_state ) ) {
					$product_state = $variation_state;
				}

				if ( 'same' === $variation_state ) {
					$product_state = get_post_meta( $product_id, '_empik_product_state_all_variants', true )
						? get_post_meta( $product_id, '_empik_product_state_all_variants', true )
						: '11';
				}
			} else {

				$product_state = get_post_meta( $product_id, '_empik_product_state_all_variants', true )
					? get_post_meta( $product_id, '_empik_product_state_all_variants', true )
					: '11';
			}
		} else {
			$product_state = get_post_meta( $product_id, '_empik_product_state_all_variants', true )
				? get_post_meta( $product_id, '_empik_product_state_all_variants', true )
				: '11';
		}

		return $product_state;
	}


	/**
	 * Gets all available variations for a variable product.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param WC_Product_Variable $wc_product The variable product object.
	 * @return array Array of available variation data.
	 */
	public function get_all_variations( $wc_product ) {

		$variation_ids  = $wc_product->get_children();
		$all_variations = array();

		if ( is_callable( '_prime_post_caches' ) ) {
			_prime_post_caches( $variation_ids );
		}

		$wc_variation = new WC_Product_Variable();

		foreach ( $variation_ids as $variation_id ) {
			$variation = wc_get_product( $variation_id );

			if ( ! $variation || ! $variation->exists() ) {
				continue;
			}

			$all_variations[] = $wc_variation->get_available_variation( $variation );
		}

		$all_variations = array_values( array_filter( $all_variations ) );

		return $all_variations;
	}
}
