<?php

namespace Inspire_Labs\Empik_Woocommerce\Offer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Exception;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Alerts_Static_Factory;
use WP_Filesystem_Direct;

/**
 * Offer_Csv
 */
class Offer_Csv {

	private $include_logistic_class = false;

	private $include_product_state = false;


	/**
	 * Creates a CSV file from array of offer models.
	 *
	 * Generates CSV with offer data including optional logistic class and product state.
	 * Saves file to WordPress uploads directory with timestamp in filename.
	 *
	 * @param Offer_Model[] $offer_model Array of offer models to export to CSV.
	 * @return string Path to the generated CSV file.
	 *
	 * @throws Exception If file operations fail.
	 */
	public function create_csv( array $offer_model ): string {

		if ( isset( $_POST['add_logistic_class'] ) && 'true' === sanitize_text_field( $_POST['add_logistic_class'] ) ) {
			$this->include_logistic_class = true;
		} else if( 'yes' === get_option( 'empik_woocommerce_set_export_logistic_class' ) ) {
            $this->include_logistic_class = true;
        }

		if ( isset( $_POST['add_product_state'] ) && 'true' === sanitize_text_field( $_POST['add_product_state'] ) ) {
			$this->include_product_state = true;
		} else if( 'yes' === get_option( 'empik_woocommerce_set_export_product_states' ) ) {
            $this->include_product_state = true;
        }

		require_once ABSPATH . 'wp-admin/includes/file.php';

		try {
			$tmp_path = wp_tempnam();
			$fp       = fopen( $tmp_path, 'w' );
			fputcsv( $fp, $this->get_columns(), ';' );

			foreach ( $offer_model as $offer_row ) {

				$scv_fields = array(
					0 => $offer_row->get_sku(),
					1 => $offer_row->get_id(),
					2 => $offer_row->get_id_type(),
					3 => $offer_row->get_quantity(),
					4 => $offer_row->get_min_quantity_alert(),
					5 => $offer_row->get_price(),
					6 => $offer_row->get_discount_price(),
					7 => $offer_row->get_discount_start_date(),
					8 => $offer_row->get_discount_end_date(),
					9 => $offer_row->get_update_delete(),
				);

				if ( $this->include_product_state ) {
					$scv_fields[] = $offer_row->get_product_state();
				} else {
					$scv_fields[] = 11;
				}

				if ( $this->include_logistic_class ) {
					$scv_fields[] = $offer_row->get_logistic_class();
				}

				fputcsv(
					$fp,
					$scv_fields,
					';'
				);
			}
			fclose( $fp );

		} catch ( Exception $e ) {
			Plugin::write_debug_log( $e->getMessage() );
		}

		$fs = new WP_Filesystem_Direct( null );

		$new_file_name = 'WooCommerce-offers-' . time() . '.csv';
		update_option( 'empik_last_export_file_name', $new_file_name );
		$new_file_path = wp_upload_dir()['basedir'] . DIRECTORY_SEPARATOR . $new_file_name;
		$fs->move( $tmp_path, $new_file_path );

		return $new_file_path;
	}


	/**
	 * Gets array of CSV column headers.
	 *
	 * Returns array of column names including optional logistic class
	 * if that feature is enabled.
	 *
	 * @return array List of column headers for CSV file.
	 */
	private function get_columns(): array {

		$column_headings = array(
			'sku',
			'product-id',
			'product-id-type',
			'quantity',
			'min-quantity-alert',
			'price',
			'discount-price',
			'discount-start-date',
			'discount-end-date',
			'update-delete',
			'state',
		);

		if ( $this->include_logistic_class ) {
			$column_headings[] = 'logistic-class';
		}

		return $column_headings;
	}
}
