<?php

namespace Inspire_Labs\Empik_Woocommerce\Api_Client;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

use Inspire_Labs\Empik_Woocommerce\Api_Client\Api_Factory;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Ids;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use WP_Error;

/**
 * Abstract_Api
 */
abstract class Abstract_Api {

    const ORDER_STATUS_WAITING_ACCEPTANCE = 'WAITING_ACCEPTANCE';
    const ORDER_STATUS_SHIPPING           = 'SHIPPING';

    /**
     * Gets the API endpoint URL.
     *
     * @return string The complete API base URL for making requests.
     */
    abstract public function get_api_url(): string;


    /**
     * Exports products
     *
     * @param string $csv_path $csv_path.
     *
     * @return Ajax_Response_Model
     */
    public function export_products( string $csv_path ): Ajax_Response_Model {

        $boundary = wp_generate_password( 24 );
        $body     = $this->prepare_file_payload(
            $csv_path,
            array(
                'shop_id' => Plugin::get_settings()
                                   ->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID ),
            ),
            $boundary
        );

        return $this->export( $body, $boundary, '/api/products/imports' );
    }

    /**
     * Export offers
     *
     * @param string $csv_path $csv_path.
     *
     * @return Ajax_Response_Model
     */
    public function export_offers( string $csv_path ): Ajax_Response_Model {

        $boundary = wp_generate_password( 24 );
        $body     = $this->prepare_file_payload(
            $csv_path,
            array(
                'shop_id'     => Plugin::get_settings()
                                       ->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID ),
                'import_mode' => 'NORMAL',
            ),
            $boundary
        );

        return $this->export( $body, $boundary, '/api/offers/imports' );
    }

    /**
     * Export
     *
     * @param string $body $body.
     * @param string $boundary $boundary.
     * @param string $endpoint $endpoint.
     *
     * @return Ajax_Response_Model
     */
    public function export( string $body, string $boundary, string $endpoint ): Ajax_Response_Model {

        $result = wp_remote_post(
            $this->get_api_url() . $endpoint,
            array(
                'headers' => array(
                    'Authorization' => Plugin::get_settings()
                                             ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
                    'content-type'  => 'multipart/form-data; boundary='
                                       . $boundary,
                ),
                'body'    => $body,
            )
        );

        $body_decoded = json_decode( wp_remote_retrieve_body( $result ), true );

        if ( isset( $body_decoded['import_id'] ) ) {
            $import_id = $body_decoded['import_id'];

            // debug get file .csv.
            if ( ! empty( get_option( 'empik_last_export_file_name' ) ) ) {

                $uploads               = wp_upload_dir();
                $last_export_file_url  = esc_url( $uploads['baseurl'] . '/' . get_option( 'empik_last_export_file_name' ) );
                $last_export_file_name = sanitize_text_field( get_option( 'empik_last_export_file_name' ) );
                delete_option( 'empik_last_export_file_name' );

                return ( new Ajax_Response_Model(
                    Ajax_Response_Model::SUCCESS,
                    sprintf(
                        "%s %s <br>%s <a download href='%s'>%s</a>",
                        esc_html__( 'Export finished. ID:', 'empik-for-woocommerce' ),
                        $import_id,
                        esc_html__( 'File:', 'empik-for-woocommerce' ),
                        $last_export_file_url,
                        $last_export_file_name
                    )
                ) );

            } else {
                return ( new Ajax_Response_Model(
                    Ajax_Response_Model::SUCCESS,
                    sprintf(
                        esc_html__( 'Export finished. ID: %s ', 'empik-for-woocommerce' ),
                        $import_id
                    )
                ) );
            }
        }

        return ( $this->get_error_from_wp_remote_result(
            $result,
            array(
                'API URL: ' => $this->get_api_url() . $endpoint,
                'API_KEY: ' => Plugin::get_settings()
                                     ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
            )
        ) );
    }

    /**
     * Get error from wp_remote result
     *
     * @param            $result $result.
     * @param array  $params_to_show $params_to_show.
     *
     * @return Ajax_Response_Model
     */
    public function get_error_from_wp_remote_result(
        $result,
        array $params_to_show = array()
    ): Ajax_Response_Model {
        if ( is_wp_error( $result ) ) {
            /**
             * @var WP_Error $result ;
             */
            return new Ajax_Response_Model(
                'error',
                $result->get_error_message()
            );
        }

        $body_decoded = json_decode( wp_remote_retrieve_body( $result ), true );

        $extra_data = ! empty( $params_to_show ) ? '   [ '
                                                   . serialize( $params_to_show )
                                                   . ' ]'
            : '';

        if ( isset( $body_decoded['message'] ) ) {
            return new Ajax_Response_Model(
                'error',
                esc_html__( 'Api Error: ', 'empik-for-woocommerce' )
                . $body_decoded['message'] . $extra_data
            );
        }

        return new Ajax_Response_Model(
            'error',
            esc_html__( 'Unknown error', 'empik-for-woocommerce' ) . $extra_data
        );
    }

    /**
     * Get list of carriers
     *
     * @return array|bool
     */
    public function sh21_get_all_carriers() {

        $result = wp_remote_get(
            $this->get_api_url() . '/api/shipping/carriers',
            array(
                'headers' => array(
                    'Authorization' => Plugin::get_settings()
                                             ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
                ),
            )
        );

        $body_decoded = json_decode( wp_remote_retrieve_body( $result ) );

        if ( ! is_object( $body_decoded )
             && ! property_exists( $body_decoded, 'carriers' )
        ) {
            return false;
        }

        if ( ! property_exists( $body_decoded, 'carriers' ) ) {
            return false;
        }

        return $body_decoded->carriers;
    }


    /**
     * Put order ship confirmation
     *
     * @param string $order_id $order_id.
     */
    public function or_24_put_order_ship_confirmation( string $order_id ): void {
        $shop_id = (int) Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID );
        $args    = array(
            'headers' => array(
                'Authorization' => Plugin::get_settings()
                                         ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
            ),
            'body'    => null,
            'method'  => 'PUT',
        );

        $result = wp_remote_request( $this->get_api_url() . "/api/orders/$order_id/ship?shop_id=$shop_id", $args );
        $result = json_decode( wp_remote_retrieve_body( $result ) );
        if ( is_object( $result ) && $result->status !== 204 ) {
            Plugin::write_debug_log( $result->message );
            return;
        }

        Plugin::write_debug_log( serialize( $result ) );
    }

    /**
     * Method API OR_21
     *
     * @param string $order_id $order_id.
     * @param array  $order_lines $order_lines.
     */
    public function or_21_order_accept( string $order_id, array $order_lines ) {
        $shop_id = (int) Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID );
        $args    = array(
            'headers' => array(
                'Authorization' => Plugin::get_settings()
                                         ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
                'content-type'  => 'application/json',
            ),
            'body'    => wp_json_encode(
                array(
                    'order_lines' =>
                        $order_lines,
                )
            ),
            'method'  => 'PUT',
        );

        $result = wp_remote_request( $this->get_api_url() . "/api/orders/$order_id/accept?shop_id=$shop_id", $args );
        $result = json_decode( wp_remote_retrieve_body( $result ) );
        if ( is_object( $result ) && $result->status !== 204 ) {
            Plugin::write_debug_log( $result->message );

            return;
        }

        Plugin::write_debug_log( serialize( $result ) );
    }


    /**
     * Updates carrier tracking information for an order via API.
     *
     * Sends PUT request to update tracking details including carrier code, name,
     * tracking URL and number for a specific order.
     *
     * @param string $order_id        Order identifier.
     * @param string $carrier_code    Carrier's code.
     * @param string $carrier_url     Tracking URL.
     * @param string $tracking_number Shipment tracking number.
     * @param string $carrier_name    Name of the carrier.
     *
     * @return Ajax_Response_Model Success response on successful update (code 204),
     *                            or error response with details on failure
     *
     * @uses wp_remote_request()
     * @uses Plugin::write_debug_log()
     */
    public function or_23_carrier_update(
        string $order_id,
        string $carrier_code,
        string $carrier_url,
        string $tracking_number,
        string $carrier_name
    ): Ajax_Response_Model {

        $shop_id = (int) Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID );
        $args    = array(
            'headers' => array(
                'Authorization' => Plugin::get_settings()
                                         ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
                'content-type'  => 'application/json',
            ),
            'body'    => wp_json_encode(
                array(
                    'carrier_code'    => $carrier_code,
                    'carrier_name'    => $carrier_name,
                    'carrier_url'     => $carrier_url,
                    'tracking_number' => $tracking_number,
                )
            ),
            'method'  => 'PUT',
        );

        $result = wp_remote_request( $this->get_api_url() . "/api/orders/$order_id/tracking?shop_id=$shop_id", $args );

        Plugin::write_debug_log( wp_json_encode( $args['body'] ) );
        Plugin::write_debug_log( wp_json_encode( $result ) );

        if ( is_array( $result ) && isset( $result['response']['code'] )
             && $result['response']['code'] == 204
        ) {
            return ( new Ajax_Response_Model(
                Ajax_Response_Model::SUCCESS,
                esc_html__(
                    'The order has been successfully updated',
                    'empik-for-woocommerce'
                )
            ) );

        }

        Plugin::write_debug_log( $this->get_error_from_wp_remote_result( $result ) );

        return ( $this->get_error_from_wp_remote_result( $result ) );
    }


    /**
     * Import
     *
     * @param string $status $status.
     *
     * @return mixed
     */
    public function import( string $status ) {
        $shop_id = (int) Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID );
        $result  = wp_remote_get(
            $this->get_api_url()
            . "/api/orders?order_state_codes=$status&paginate=false&shop_id=$shop_id",
            array(
                'headers' => array(
                    'Authorization' => Plugin::get_settings()
                                             ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
                ),
            )
        );

        return json_decode( wp_remote_retrieve_body( $result ) );
    }

    /**
     * Connection test
     *
     * @return bool
     */
    public function connection_test(): bool {

        $shop_id = (int) Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_SHOP_ID );

        $result = wp_remote_get(
            $this->get_api_url() . "/api/orders?shop_id=$shop_id",
            array(
                'headers' => array(
                    'Authorization' => Plugin::get_settings()
                                             ->get_option( Settings_Ids::SETTINGS_ID_API_KEY ),
                ),
            )
        );

        /*if( 'yes' === Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_DEBUG_EMPIK ) ) {*/
        if ( true ) {
            \wc_get_logger()->debug( 'Test połączenia Empik: ', array( 'source' => 'empik-connection' ) );
            if ( isset( $result['response'] ) ) {
                \wc_get_logger()->debug( print_r( $result['response'], true ), array( 'source' => 'empik-connection' ) );
            }
        }

        $result = json_decode( wp_remote_retrieve_body( $result ) );

        return is_object( $result ) && property_exists( $result, 'orders' );
    }


    /**
     * Prepare payload
     *
     * @param string   $local_file $local_file.
     * @param array    $post_fields $post_fields.
     * @param        $boundary $boundary.
     *
     * @return string
     */
    protected function prepare_file_payload(
        string $local_file,
        array $post_fields,
        $boundary
    ): string {
        $payload = '';

        foreach ( $post_fields as $name => $value ) {
            $payload .= '--' . $boundary;
            $payload .= "\r\n";
            $payload .= 'Content-Disposition: form-data; name="' . $name .
                        '"' . "\r\n\r\n";
            $payload .= $value;
            $payload .= "\r\n";
        }

        $payload .= '--' . $boundary;
        $payload .= "\r\n";
        $payload .= 'Content-Disposition: form-data; name="' . 'file' .
                    '"; filename="' . basename( $local_file ) . '"' . "\r\n";
        $payload .= 'Content-Type: application/CSV' . "\r\n";
        $payload .= "\r\n";
        $payload .= file_get_contents( $local_file );
        $payload .= "\r\n";
        $payload .= '--' . $boundary . '--';

        return $payload;
    }


    /**
     * Get product states
     *
     * @return array|mixed
     */
    public function of61_get_product_states() {

        $api_key        = Plugin::get_settings()->get_option( Settings_Ids::SETTINGS_ID_API_KEY );

        if( empty( $api_key ) ) {
            return false;
        }

        $cache_key      = 'empik_of61_product_states';
        $product_states = get_transient( $cache_key );

        if ( empty( $product_states ) || ! is_array( $product_states ) ) {

            $result = wp_remote_get(
                $this->get_api_url() . '/api/offers/states',
                array(
                    'headers' => array(
                        'Authorization' => $api_key,
                    ),
                )
            );

            $body_decoded = json_decode( wp_remote_retrieve_body( $result ) );

            if ( ! $body_decoded || is_wp_error( $result ) ) {
                \wc_get_logger()->debug(
                    'API response, some error, method of61',
                    array( 'source' => 'empik-of61-debug' )
                );
                \wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-of61-debug' ) );

                return false;
            }

            if ( ! is_object( $body_decoded ) ) {

                \wc_get_logger()->debug(
                    'API response, not is_object, method of61',
                    array( 'source' => 'empik-of61-debug' )
                );
                \wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-of61-debug' ) );

                return false;
            }

            if ( ! property_exists( $body_decoded, 'offer_states' ) ) {

                \wc_get_logger()->debug(
                    'API response, no property_exists, method of61',
                    array( 'source' => 'empik-of61-debug' )
                );
                \wc_get_logger()->debug( print_r( $result, true ), array( 'source' => 'empik-of61-debug' ) );

                return false;
            }

            if ( ! empty( $body_decoded->offer_states ) && is_array( $body_decoded->offer_states ) ) {
                $product_states = array();
                foreach ( $body_decoded->offer_states as $offer_state ) {
                    if ( ! empty( $offer_state->label ) && ! empty( $offer_state->code ) ) {
                        $product_states[ $offer_state->code ] = $offer_state->label;
                    }
                }

                set_transient( $cache_key, $product_states, 3600 * 24 );
            }
        }

        return $product_states;
    }
}
